import { pgTable, text, integer, boolean, timestamp, jsonb } from "drizzle-orm/pg-core";
import { createInsertSchema, createSelectSchema } from "drizzle-zod";
import { z } from "zod";

export const themes = pgTable("themes", {
  id: integer("id").primaryKey().generatedAlwaysAsIdentity(),
  name: text("name").notNull(),
  displayName: text("display_name").notNull(),
  description: text("description"),
  isDefault: boolean("is_default").default(false),
  primaryColor: text("primary_color").default("#3b82f6"),
  secondaryColor: text("secondary_color").default("#64748b"),
  accentColor: text("accent_color").default("#10b981"),
  backgroundColor: text("background_color").default("#ffffff"),
  surfaceColor: text("surface_color").default("#f8fafc"),
  textColor: text("text_color").default("#1e293b"),
  mutedTextColor: text("muted_text_color").default("#64748b"),
  borderColor: text("border_color").default("#e2e8f0"),
  gradientFrom: text("gradient_from"),
  gradientTo: text("gradient_to"),
  gradientDirection: text("gradient_direction").default("to-r"),
  fontFamily: text("font_family").default("Inter"),
  borderRadius: text("border_radius").default("8px"),
  shadow: text("shadow").default("0 1px 3px 0 rgba(0, 0, 0, 0.1)"),
  customCSS: text("custom_css"),
  formStyles: jsonb("form_styles").$type<{
    inputStyle: 'modern' | 'classic' | 'minimal' | 'outlined' | 'filled';
    buttonStyle: 'rounded' | 'square' | 'pill' | 'ghost';
    cardStyle: 'elevated' | 'outlined' | 'flat' | 'glass';
    layout: 'vertical' | 'horizontal' | 'grid' | 'masonry';
  }>(),
  animations: jsonb("animations").$type<{
    enabled: boolean;
    duration: number;
    easing: string;
    hoverEffects: boolean;
    transitions: boolean;
  }>(),
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const paymentGateways = pgTable("payment_gateways", {
  id: integer("id").primaryKey().generatedAlwaysAsIdentity(),
  name: text("name").notNull(),
  displayName: text("display_name").notNull(),
  provider: text("provider").notNull(), // stripe, paypal, flutterwave, etc.
  isActive: boolean("is_active").default(false),
  isSandbox: boolean("is_sandbox").default(true),
  config: jsonb("config").$type<{
    publicKey?: string;
    secretKey?: string;
    webhookSecret?: string;
    merchantId?: string;
    apiKey?: string;
    baseUrl?: string;
    [key: string]: any;
  }>(),
  supportedCurrencies: text("supported_currencies").array().default([]),
  fees: jsonb("fees").$type<{
    percentage: number;
    fixed: number;
    currency: string;
  }>(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const formTemplates = pgTable("form_templates", {
  id: integer("id").primaryKey().generatedAlwaysAsIdentity(),
  name: text("name").notNull(),
  displayName: text("display_name").notNull(),
  description: text("description"),
  category: text("category").notNull(), // sports, business, education, etc.
  themeId: integer("theme_id").references(() => themes.id),
  fields: jsonb("fields").$type<Array<{
    id: string;
    type: string;
    label: string;
    placeholder?: string;
    required: boolean;
    validation?: any;
    options?: string[];
  }>>(),
  layout: jsonb("layout").$type<{
    columns: number;
    spacing: string;
    alignment: string;
    sections: Array<{
      title: string;
      fields: string[];
    }>;
  }>(),
  notifications: jsonb("notifications").$type<{
    emailEnabled: boolean;
    emailTemplate: string;
    recipients: string[];
    webhookUrl?: string;
  }>(),
  isPublic: boolean("is_public").default(false),
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertThemeSchema = createInsertSchema(themes);
export const selectThemeSchema = createSelectSchema(themes);
export type InsertTheme = z.infer<typeof insertThemeSchema>;
export type SelectTheme = z.infer<typeof selectThemeSchema>;

export const insertPaymentGatewaySchema = createInsertSchema(paymentGateways);
export const selectPaymentGatewaySchema = createSelectSchema(paymentGateways);
export type InsertPaymentGateway = z.infer<typeof insertPaymentGatewaySchema>;
export type SelectPaymentGateway = z.infer<typeof selectPaymentGatewaySchema>;

export const insertFormTemplateSchema = createInsertSchema(formTemplates);
export const selectFormTemplateSchema = createSelectSchema(formTemplates);
export type InsertFormTemplate = z.infer<typeof insertFormTemplateSchema>;
export type SelectFormTemplate = z.infer<typeof selectFormTemplateSchema>;