export const SPORTS = [
  { value: "football", label: "Football" },
  { value: "rugby", label: "Rugby" },
  { value: "cricket", label: "Cricket" },
  { value: "baseball", label: "Baseball" },
  { value: "basketball", label: "Basketball" },
  { value: "hockey", label: "Hockey" },
  { value: "netball", label: "Netball" },
  { value: "soccer", label: "Soccer" },
  { value: "badminton", label: "Badminton" },
  { value: "volleyball", label: "Volleyball" },
] as const;

export type SportType = typeof SPORTS[number]["value"];

// Sport categories for recommendation engine
export const SPORT_CATEGORIES = {
  team: ["football", "rugby", "cricket", "baseball", "basketball", "hockey", "netball", "soccer", "volleyball"],
  individual: ["badminton"],
  contact: ["football", "rugby", "hockey"],
  nonContact: ["cricket", "baseball", "basketball", "netball", "soccer", "badminton", "volleyball"],
  fieldBased: ["football", "rugby", "soccer", "hockey"],
  courtBased: ["basketball", "netball", "badminton", "volleyball"],
  pitchBased: ["cricket", "baseball"]
} as const;

// Complexity levels for onboarding recommendations
export const SPORT_COMPLEXITY = {
  beginner: ["badminton", "basketball", "netball"],
  intermediate: ["soccer", "volleyball", "hockey", "baseball"],
  advanced: ["football", "rugby", "cricket"]
} as const;

// Priority features by sport for onboarding
export const SPORT_PRIORITIES = {
  football: ["roster", "formations", "analytics", "substitutions"],
  rugby: ["roster", "formations", "analytics", "lineouts"],
  cricket: ["players", "batting", "bowling", "statistics"],
  baseball: ["roster", "batting", "pitching", "statistics"],
  basketball: ["roster", "rotations", "analytics", "substitutions"],
  hockey: ["roster", "formations", "analytics", "penalties"],
  netball: ["roster", "positions", "rotations", "analytics"],
  soccer: ["roster", "formations", "analytics", "tactics"],
  badminton: ["players", "tournaments", "rankings", "matches"],
  volleyball: ["roster", "rotations", "analytics", "substitutions"]
} as const;

// Sport-specific features mapping
export const SPORT_FEATURES = {
  football: {
    positions: ["Quarterback", "Running Back", "Wide Receiver", "Tight End", "Offensive Line", "Defensive Line", "Linebacker", "Safety", "Cornerback"],
    formations: ["I-Formation", "Shotgun", "Pistol", "Wildcat"],
    teamSize: 11,
    substitutions: "unlimited",
    scoreTypes: ["Touchdown", "Field Goal", "Safety", "Extra Point"]
  },
  rugby: {
    positions: ["Prop", "Hooker", "Lock", "Flanker", "Number 8", "Scrum Half", "Fly Half", "Center", "Wing", "Fullback"],
    formations: ["Scrum", "Lineout", "Ruck", "Maul"],
    teamSize: 15,
    substitutions: "limited",
    scoreTypes: ["Try", "Conversion", "Penalty", "Drop Goal"]
  },
  cricket: {
    positions: ["Batsman", "Bowler", "Wicket Keeper", "All-rounder"],
    formations: ["Fielding Positions"],
    teamSize: 11,
    substitutions: "limited",
    scoreTypes: ["Runs", "Wickets", "Extras"]
  },
  baseball: {
    positions: ["Pitcher", "Catcher", "First Base", "Second Base", "Third Base", "Shortstop", "Left Field", "Center Field", "Right Field"],
    formations: ["Infield", "Outfield"],
    teamSize: 9,
    substitutions: "unlimited",
    scoreTypes: ["Run", "Home Run", "RBI"]
  },
  basketball: {
    positions: ["Point Guard", "Shooting Guard", "Small Forward", "Power Forward", "Center"],
    formations: ["Man-to-Man", "Zone Defense", "Pick and Roll"],
    teamSize: 5,
    substitutions: "unlimited",
    scoreTypes: ["Field Goal", "Free Throw", "Three Pointer"]
  },
  hockey: {
    positions: ["Goalkeeper", "Defender", "Midfielder", "Forward"],
    formations: ["4-3-3", "3-5-2", "4-4-2"],
    teamSize: 11,
    substitutions: "limited",
    scoreTypes: ["Goal", "Penalty Corner", "Penalty Stroke"]
  },
  netball: {
    positions: ["Goal Shooter", "Goal Attack", "Wing Attack", "Center", "Wing Defense", "Goal Defense", "Goal Keeper"],
    formations: ["Center Pass", "Defensive Formation"],
    teamSize: 7,
    substitutions: "limited",
    scoreTypes: ["Goal"]
  },
  soccer: {
    positions: ["Goalkeeper", "Defender", "Midfielder", "Forward"],
    formations: ["4-4-2", "4-3-3", "3-5-2", "4-2-3-1"],
    teamSize: 11,
    substitutions: "limited",
    scoreTypes: ["Goal", "Penalty", "Own Goal"]
  },
  badminton: {
    positions: ["Singles", "Doubles"],
    formations: ["Court Positions"],
    teamSize: 1,
    substitutions: "none",
    scoreTypes: ["Point", "Rally"]
  },
  volleyball: {
    positions: ["Setter", "Outside Hitter", "Middle Blocker", "Opposite Hitter", "Libero"],
    formations: ["6-2", "5-1", "4-2"],
    teamSize: 6,
    substitutions: "limited",
    scoreTypes: ["Point", "Spike", "Block", "Ace"]
  }
} as const;

// Get sport-specific positions
export function getSportPositions(sport: SportType): string[] {
  return [...(SPORT_FEATURES[sport]?.positions || [])];
}

// Get sport-specific formations
export function getSportFormations(sport: SportType): string[] {
  return [...(SPORT_FEATURES[sport]?.formations || [])];
}

// Get sport-specific team size
export function getSportTeamSize(sport: SportType): number {
  return SPORT_FEATURES[sport]?.teamSize || 11;
}

// Check if sport allows unlimited substitutions
export function hasUnlimitedSubstitutions(sport: SportType): boolean {
  return SPORT_FEATURES[sport]?.substitutions === "unlimited";
}

// Get sport-specific score types
export function getSportScoreTypes(sport: SportType): string[] {
  return [...(SPORT_FEATURES[sport]?.scoreTypes || [])];
}

// Smart onboarding helper functions
export function getSportCategory(sport: SportType): string[] {
  const categories: string[] = [];
  Object.entries(SPORT_CATEGORIES).forEach(([category, sports]) => {
    if ((sports as readonly SportType[]).includes(sport)) {
      categories.push(category);
    }
  });
  return categories;
}

export function getSportComplexity(sport: SportType): keyof typeof SPORT_COMPLEXITY {
  for (const [complexity, sports] of Object.entries(SPORT_COMPLEXITY)) {
    if ((sports as readonly SportType[]).includes(sport)) {
      return complexity as keyof typeof SPORT_COMPLEXITY;
    }
  }
  return "intermediate";
}

export function getSportPriorities(sport: SportType): string[] {
  return [...(SPORT_PRIORITIES[sport] || [])];
}

export function isTeamSport(sport: SportType): boolean {
  return (SPORT_CATEGORIES.team as readonly SportType[]).includes(sport);
}

export function isContactSport(sport: SportType): boolean {
  return (SPORT_CATEGORIES.contact as readonly SportType[]).includes(sport);
}

export function getSportRecommendations(sport: SportType) {
  const complexity = getSportComplexity(sport);
  const categories = getSportCategory(sport);
  const priorities = getSportPriorities(sport);
  const isTeam = isTeamSport(sport);
  const isContact = isContactSport(sport);
  
  return {
    complexity,
    categories,
    priorities,
    isTeam,
    isContact,
    teamSize: getSportTeamSize(sport),
    positions: getSportPositions(sport),
    formations: getSportFormations(sport),
    hasUnlimitedSubs: hasUnlimitedSubstitutions(sport)
  };
}