import { MailService } from '@sendgrid/mail';

if (!process.env.SENDGRID_API_KEY) {
  console.log("SENDGRID_API_KEY environment variable not set. Email functionality will be disabled.");
} else {
  const mailService = new MailService();
  mailService.setApiKey(process.env.SENDGRID_API_KEY);
}

interface ConfirmationEmailParams {
  to: string;
  firstName: string;
  verificationToken: string;
  baseUrl: string;
}

export async function sendConfirmationEmail(params: ConfirmationEmailParams): Promise<boolean> {
  if (!process.env.SENDGRID_API_KEY) {
    console.log("Confirmation email would be sent to:", params.to);
    console.log("Verification link:", `${params.baseUrl}/verify-email?token=${params.verificationToken}`);
    return false;
  }

  try {
    const mailService = new MailService();
    mailService.setApiKey(process.env.SENDGRID_API_KEY);
    
    const verificationLink = `${params.baseUrl}/verify-email?token=${params.verificationToken}`;
    
    const htmlContent = `
      <!DOCTYPE html>
      <html>
      <head>
        <meta charset="utf-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Email Verification - SportsPro</title>
        <style>
          body { 
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif; 
            line-height: 1.6; 
            color: #666; 
            margin: 0; 
            padding: 0; 
            background-color: #f5f7fa;
          }
          .container { 
            max-width: 600px; 
            margin: 0 auto; 
            background: white;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
          }
          .header { 
            background: linear-gradient(135deg, #1e3a8a 0%, #3730a3 100%); 
            color: white; 
            padding: 60px 40px; 
            text-align: center; 
            position: relative;
          }
          .header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-image: 
              radial-gradient(circle at 25% 25%, rgba(255,255,255,0.1) 2px, transparent 2px),
              radial-gradient(circle at 75% 75%, rgba(255,255,255,0.1) 2px, transparent 2px);
            background-size: 50px 50px;
          }
          .logo { 
            position: relative;
            z-index: 1;
            display: inline-block;
            background: white;
            color: #1e3a8a;
            padding: 12px 24px;
            border-radius: 8px;
            font-size: 24px;
            font-weight: bold;
            margin-bottom: 15px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.2);
          }
          .tagline {
            position: relative;
            z-index: 1;
            font-size: 14px;
            opacity: 0.9;
          }
          .content { 
            padding: 60px 50px; 
            background: #ffffff; 
          }
          .greeting {
            font-size: 32px;
            font-weight: 400;
            color: #374151;
            margin-bottom: 30px;
          }
          .message {
            font-size: 16px;
            margin-bottom: 25px;
            color: #666;
            line-height: 1.7;
          }
          .button { 
            background: #10b981; 
            color: white; 
            padding: 18px 40px; 
            text-decoration: none; 
            border-radius: 8px; 
            display: inline-block; 
            font-weight: 600;
            margin: 30px 0;
            font-size: 16px;
            box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
          }
          .button:hover {
            background: #059669;
          }
          .warning { 
            background: #fff3cd; 
            border-left: 4px solid #f59e0b;
            padding: 20px; 
            margin: 30px 0; 
            color: #92400e;
            border-radius: 0 8px 8px 0;
          }
          .mandatory {
            background: #f3f4f6;
            padding: 20px;
            margin: 30px 0;
            color: #374151;
            border-radius: 8px;
            border-left: 4px solid #6b7280;
          }
          .support-section {
            margin-top: 40px;
            padding-top: 30px;
            border-top: 1px solid #e5e7eb;
          }
          .support-link {
            color: #10b981;
            text-decoration: none;
            font-weight: 500;
          }
          .footer { 
            text-align: center; 
            padding: 40px; 
            color: #6b7280; 
            font-size: 14px; 
            background: #f9fafb;
            border-top: 1px solid #e5e7eb;
          }
          .footer-signature {
            font-weight: 600;
            color: #374151;
            margin-bottom: 20px;
          }
          .app-buttons {
            margin-top: 30px;
            display: flex;
            justify-content: center;
            gap: 15px;
          }
          .app-button {
            display: inline-block;
            height: 40px;
            background: #000;
            color: white;
            padding: 10px 20px;
            border-radius: 6px;
            text-decoration: none;
            font-size: 12px;
            font-weight: 500;
          }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="header">
            <div class="logo">SP</div>
            <div class="tagline">SportsPro</div>
          </div>
          <div class="content">
            <div class="greeting">Welcome ${params.firstName},</div>
            
            <div class="message">Your account has been successfully created!</div>
            
            <div class="message">To complete the creation of your account, please click on the verification button below.</div>
            
            <div class="warning">
              This link is valid for 48 hours. After this period, you will have to reconnect to receive another email.
            </div>
            
            <div class="mandatory">
              This verification is mandatory to continue using SportsPro.
            </div>
            
            <div style="text-align: center;">
              <a href="${verificationLink}" class="button">Verify my email</a>
            </div>
            
            <div class="support-section">
              <div class="message">
                If you encounter any problems when verifying your account, please feel free to contact our customer service by <a href="mailto:support@sportspro.com" class="support-link">clicking here</a>.
              </div>
              
              <div class="message" style="margin-top: 30px;">
                See you soon!
              </div>
              <div class="footer-signature">The SportsPro team</div>
            </div>
          </div>
          <div class="footer">
            <div class="app-buttons">
              <a href="#" class="app-button">Available on the App Store</a>
              <a href="#" class="app-button">Get it on Google Play</a>
            </div>
            <p style="margin-top: 30px; font-size: 12px; color: #9ca3af;">
              This email was sent to ${params.to}. If you didn't create an account with SportsPro, please ignore this email.
            </p>
          </div>
        </div>
      </body>
      </html>
    `;

    await mailService.send({
      to: params.to,
      from: 'noreply@sportspro.com',
      subject: 'Verify your SportsPro account',
      html: htmlContent,
    });

    return true;
  } catch (error) {
    console.error('SendGrid confirmation email error:', error);
    return false;
  }
}