import React from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { 
  Trophy, 
  Users, 
  Target, 
  Shield, 
  BarChart3, 
  Smartphone, 
  Zap, 
  CheckCircle, 
  Star,
  Download,
  Mail,
  Calendar
} from "lucide-react";
import dashboardImage from "@assets/generated_images/Sports_management_dashboard_interface_a30de65d.png";
import analyticsImage from "@assets/generated_images/Sports_analytics_visualization_dashboard_5dee7bc7.png";
import teamManagementImage from "@assets/generated_images/Team_management_interface_design_f2ee08c3.png";

export default function Proposal() {
  const features = [
    {
      icon: <Trophy className="h-6 w-6" />,
      title: "Team Management",
      items: [
        "Dynamic team creation across multiple sports",
        "Interactive drag-and-drop roster management", 
        "Comprehensive player profiles",
        "Coach assignment and role management",
        "Formation support for team sports"
      ]
    },
    {
      icon: <BarChart3 className="h-6 w-6" />,
      title: "Performance Analytics",
      items: [
        "AI-powered performance insights",
        "One-click video highlight generation",
        "Biometric data integration", 
        "Statistical tracking and trends",
        "Real-time dashboard visualization"
      ]
    },
    {
      icon: <Shield className="h-6 w-6" />,
      title: "Security & Privacy",
      items: [
        "Multi-role access control",
        "Session-based authentication",
        "GDPR-compliant data handling",
        "Granular privacy controls",
        "Comprehensive audit trails"
      ]
    },
    {
      icon: <Target className="h-6 w-6" />,
      title: "Business Management",
      items: [
        "Organization-based subscriptions",
        "Multiple payment gateway support",
        "Automated fee collection",
        "Financial reporting and analytics",
        "Membership tracking"
      ]
    },
    {
      icon: <Smartphone className="h-6 w-6" />,
      title: "Modern Interface",
      items: [
        "Mobile-first responsive design",
        "18 professional theme options",
        "WCAG accessibility compliance",
        "Intuitive navigation",
        "Customizable layouts"
      ]
    },
    {
      icon: <Zap className="h-6 w-6" />,
      title: "Smart Features",
      items: [
        "AI-powered onboarding recommendations",
        "Smart setup scoring system",
        "Automated notifications",
        "Dynamic content management",
        "Real-time synchronization"
      ]
    }
  ];

  const stats = [
    { number: "10+", label: "Sports Supported" },
    { number: "18", label: "Professional Themes" },
    { number: "80%", label: "Admin Time Savings" },
    { number: "100%", label: "Mobile Responsive" }
  ];

  const techStack = [
    {
      category: "Frontend",
      technologies: ["React 18 + TypeScript", "Vite Build System", "TanStack Query", "Tailwind CSS"]
    },
    {
      category: "Backend", 
      technologies: ["Node.js 20.x", "Express.js", "PostgreSQL", "Drizzle ORM"]
    },
    {
      category: "Security",
      technologies: ["Session Management", "Role-Based Access", "Data Encryption", "GDPR Compliance"]
    },
    {
      category: "Deployment",
      technologies: ["Production Ready", "Nginx + PM2", "Auto Scaling", "Performance Optimized"]
    }
  ];

  const sports = {
    team: ["Football (American)", "Soccer (Football)", "Rugby", "Basketball", "Hockey", "Netball", "Volleyball"],
    individual: ["Cricket", "Baseball", "Badminton", "Tennis", "Athletics"]
  };

  const timeline = [
    {
      phase: "Phase 1: Setup",
      duration: "Weeks 1-2",
      tasks: ["Platform deployment", "Initial configuration", "Team setup", "Basic training"]
    },
    {
      phase: "Phase 2: Rollout", 
      duration: "Weeks 3-4",
      tasks: ["Advanced features", "Custom configuration", "System integration", "Staff training"]
    },
    {
      phase: "Phase 3: Optimization",
      duration: "Weeks 5-6", 
      tasks: ["Performance tuning", "User feedback", "Feature enhancement", "Success metrics"]
    },
    {
      phase: "Phase 4: Production",
      duration: "Week 7+",
      tasks: ["Full utilization", "Ongoing support", "Continuous improvement", "Success evaluation"]
    }
  ];

  const advantages = [
    {
      title: "Technical Excellence",
      items: ["Modern, scalable architecture", "Enterprise-grade security", "Mobile-optimized experience", "API-first design"]
    },
    {
      title: "Business Value", 
      items: ["Rapid deployment capability", "Customizable for any sport", "Cost-effective solution", "Integration-ready platform"]
    },
    {
      title: "Innovation Features",
      items: ["AI-powered analytics", "Automated video highlights", "Biometric integration", "Smart recommendations"]
    }
  ];

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-gradient-to-r from-blue-900 to-blue-600 text-white py-16">
        <div className="container mx-auto px-4 text-center">
          <h1 className="text-5xl font-bold mb-6">SportsPro</h1>
          <p className="text-xl opacity-90 max-w-3xl mx-auto">
            Enterprise Sports Management Platform - Transforming Sports Management Through Technology
          </p>
        </div>
      </div>

      <div className="container mx-auto px-4 py-12">
        
        {/* Executive Summary */}
        <Card className="mb-12">
          <CardHeader>
            <CardTitle className="text-3xl text-blue-900">Executive Summary</CardTitle>
          </CardHeader>
          <CardContent>
            <p className="text-lg text-gray-700 mb-8">
              SportsPro is a comprehensive, enterprise-grade sports management platform designed specifically for 
              professional sports organizations. Our solution provides cutting-edge team management capabilities, 
              advanced analytics, and streamlined operations that transform how sports organizations manage their 
              teams, players, and performance data.
            </p>
            
            <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
              {stats.map((stat, index) => (
                <div key={index} className="text-center p-6 bg-blue-50 rounded-lg">
                  <div className="text-3xl font-bold text-blue-600 mb-2">{stat.number}</div>
                  <div className="text-gray-600">{stat.label}</div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        {/* Platform Overview with Images */}
        <Card className="mb-12">
          <CardHeader>
            <CardTitle className="text-3xl text-blue-900">Platform Overview</CardTitle>
          </CardHeader>
          <CardContent className="space-y-12">
            
            <div className="text-center">
              <h3 className="text-2xl font-semibold mb-4">Dashboard Interface</h3>
              <img 
                src={dashboardImage} 
                alt="SportsPro Dashboard Interface"
                className="w-full max-w-4xl mx-auto rounded-lg shadow-lg mb-4"
              />
              <p className="text-gray-600 italic">
                Modern, intuitive dashboard showing team management, player analytics, and performance metrics
              </p>
            </div>

            <div className="text-center">
              <h3 className="text-2xl font-semibold mb-4">Advanced Analytics</h3>
              <img 
                src={analyticsImage} 
                alt="Sports Analytics Dashboard"
                className="w-full max-w-4xl mx-auto rounded-lg shadow-lg mb-4"
              />
              <p className="text-gray-600 italic">
                Comprehensive analytics and data visualization for performance insights
              </p>
            </div>

            <div className="text-center">
              <h3 className="text-2xl font-semibold mb-4">Team Management</h3>
              <img 
                src={teamManagementImage} 
                alt="Team Management Interface"
                className="w-full max-w-4xl mx-auto rounded-lg shadow-lg mb-4"
              />
              <p className="text-gray-600 italic">
                Professional team and player management with roster organization tools
              </p>
            </div>
          </CardContent>
        </Card>

        {/* Core Features */}
        <Card className="mb-12">
          <CardHeader>
            <CardTitle className="text-3xl text-blue-900">Core Platform Features</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
              {features.map((feature, index) => (
                <div key={index} className="bg-gray-50 p-6 rounded-lg border-l-4 border-blue-500">
                  <div className="flex items-center mb-4">
                    <div className="text-blue-600 mr-3">{feature.icon}</div>
                    <h4 className="text-lg font-semibold">{feature.title}</h4>
                  </div>
                  <ul className="space-y-2">
                    {feature.items.map((item, itemIndex) => (
                      <li key={itemIndex} className="flex items-start">
                        <CheckCircle className="h-4 w-4 text-green-500 mr-2 mt-0.5 flex-shrink-0" />
                        <span className="text-sm text-gray-700">{item}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        {/* Technical Architecture */}
        <Card className="mb-12">
          <CardHeader>
            <CardTitle className="text-3xl text-blue-900">Technical Architecture</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              {techStack.map((tech, index) => (
                <div key={index} className="bg-yellow-50 p-6 rounded-lg text-center">
                  <h4 className="font-semibold text-lg mb-3">{tech.category}</h4>
                  {tech.technologies.map((technology, techIndex) => (
                    <p key={techIndex} className="text-sm text-gray-700 mb-1">{technology}</p>
                  ))}
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        {/* Multi-Sport Support */}
        <Card className="mb-12">
          <CardHeader>
            <CardTitle className="text-3xl text-blue-900">Multi-Sport Capabilities</CardTitle>
          </CardHeader>
          <CardContent>
            <p className="text-lg text-gray-700 mb-6">
              SportsPro supports comprehensive management across multiple sports disciplines:
            </p>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
              <div className="bg-gray-50 p-6 rounded-lg border-l-4 border-blue-500">
                <h4 className="text-lg font-semibold mb-4 flex items-center">
                  <Users className="h-5 w-5 mr-2" />
                  Team Sports
                </h4>
                <ul className="space-y-2">
                  {sports.team.map((sport, index) => (
                    <li key={index} className="flex items-center">
                      <CheckCircle className="h-4 w-4 text-green-500 mr-2" />
                      <span className="text-sm">{sport}</span>
                    </li>
                  ))}
                </ul>
              </div>
              <div className="bg-gray-50 p-6 rounded-lg border-l-4 border-blue-500">
                <h4 className="text-lg font-semibold mb-4 flex items-center">
                  <Target className="h-5 w-5 mr-2" />
                  Individual Sports
                </h4>
                <ul className="space-y-2">
                  {sports.individual.map((sport, index) => (
                    <li key={index} className="flex items-center">
                      <CheckCircle className="h-4 w-4 text-green-500 mr-2" />
                      <span className="text-sm">{sport}</span>
                    </li>
                  ))}
                </ul>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Competitive Advantages */}
        <Card className="mb-12">
          <CardHeader>
            <CardTitle className="text-3xl text-blue-900">Competitive Advantages</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
              {advantages.map((advantage, index) => (
                <div key={index} className="bg-purple-50 p-6 rounded-lg border-l-4 border-purple-500">
                  <h4 className="text-lg font-semibold mb-4">{advantage.title}</h4>
                  <ul className="space-y-2">
                    {advantage.items.map((item, itemIndex) => (
                      <li key={itemIndex} className="flex items-start">
                        <CheckCircle className="h-4 w-4 text-green-500 mr-2 mt-0.5 flex-shrink-0" />
                        <span className="text-sm text-gray-700">{item}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        {/* Implementation Timeline */}
        <Card className="mb-12">
          <CardHeader>
            <CardTitle className="text-3xl text-blue-900">Implementation Timeline</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              {timeline.map((phase, index) => (
                <div key={index} className="bg-gray-50 p-6 rounded-lg border-t-4 border-blue-500">
                  <h4 className="font-semibold text-lg mb-2">{phase.phase}</h4>
                  <p className="text-blue-600 font-medium mb-3">{phase.duration}</p>
                  <ul className="space-y-1">
                    {phase.tasks.map((task, taskIndex) => (
                      <li key={taskIndex} className="flex items-start">
                        <CheckCircle className="h-3 w-3 text-green-500 mr-2 mt-1 flex-shrink-0" />
                        <span className="text-sm text-gray-700">{task}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        {/* ROI Benefits */}
        <Card className="mb-12">
          <CardHeader>
            <CardTitle className="text-3xl text-blue-900">Return on Investment</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-6">
              <div className="bg-green-50 p-6 rounded-lg border-l-4 border-green-500">
                <h4 className="font-semibold text-lg mb-2">Operational Efficiency</h4>
                <p className="text-gray-700">60-80% reduction in administrative tasks through automation and streamlined workflows.</p>
              </div>
              
              <div className="bg-green-50 p-6 rounded-lg border-l-4 border-green-500">
                <h4 className="font-semibold text-lg mb-2">Data-Driven Performance</h4>
                <p className="text-gray-700">Advanced analytics enable strategic decision-making and measurable performance improvements.</p>
              </div>
              
              <div className="bg-green-50 p-6 rounded-lg border-l-4 border-green-500">
                <h4 className="font-semibold text-lg mb-2">Cost Savings</h4>
                <p className="text-gray-700">Reduced operational overhead and improved resource allocation through efficient management tools.</p>
              </div>
              
              <div className="bg-green-50 p-6 rounded-lg border-l-4 border-green-500">
                <h4 className="font-semibold text-lg mb-2">Scalable Growth</h4>
                <p className="text-gray-700">Platform grows with your organization, supporting expansion without additional infrastructure investment.</p>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* CMS Features */}
        <Card className="mb-12">
          <CardHeader>
            <CardTitle className="text-3xl text-blue-900">Integrated Content Management System</CardTitle>
          </CardHeader>
          <CardContent>
            <p className="text-lg text-gray-700 mb-6">
              Our built-in CMS empowers organizations to maintain their digital presence:
            </p>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
              <div className="bg-gray-50 p-6 rounded-lg border-l-4 border-blue-500">
                <h4 className="text-lg font-semibold mb-4">Dynamic Content</h4>
                <ul className="space-y-2">
                  <li className="flex items-center">
                    <CheckCircle className="h-4 w-4 text-green-500 mr-2" />
                    <span className="text-sm">Real-time landing page editing</span>
                  </li>
                  <li className="flex items-center">
                    <CheckCircle className="h-4 w-4 text-green-500 mr-2" />
                    <span className="text-sm">Feature management</span>
                  </li>
                  <li className="flex items-center">
                    <CheckCircle className="h-4 w-4 text-green-500 mr-2" />
                    <span className="text-sm">Testimonial system</span>
                  </li>
                  <li className="flex items-center">
                    <CheckCircle className="h-4 w-4 text-green-500 mr-2" />
                    <span className="text-sm">Statistics dashboard</span>
                  </li>
                </ul>
              </div>
              <div className="bg-gray-50 p-6 rounded-lg border-l-4 border-blue-500">
                <h4 className="text-lg font-semibold mb-4">Real-Time Updates</h4>
                <ul className="space-y-2">
                  <li className="flex items-center">
                    <CheckCircle className="h-4 w-4 text-green-500 mr-2" />
                    <span className="text-sm">Instant content updates</span>
                  </li>
                  <li className="flex items-center">
                    <CheckCircle className="h-4 w-4 text-green-500 mr-2" />
                    <span className="text-sm">No deployment required</span>
                  </li>
                  <li className="flex items-center">
                    <CheckCircle className="h-4 w-4 text-green-500 mr-2" />
                    <span className="text-sm">Live preview capability</span>
                  </li>
                  <li className="flex items-center">
                    <CheckCircle className="h-4 w-4 text-green-500 mr-2" />
                    <span className="text-sm">Version control</span>
                  </li>
                </ul>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Call to Action */}
        <Card className="bg-gradient-to-r from-green-600 to-green-500 text-white">
          <CardContent className="text-center py-12">
            <h2 className="text-4xl font-bold mb-6">Ready to Transform Your Sports Management?</h2>
            <p className="text-xl opacity-90 mb-8 max-w-3xl mx-auto">
              Join leading sports organizations already using SportsPro to streamline operations, 
              enhance performance, and achieve competitive advantage.
            </p>
            
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Button className="bg-white text-green-600 hover:bg-gray-100">
                <Calendar className="h-4 w-4 mr-2" />
                Schedule a Demo
              </Button>
              <Button variant="outline" className="border-white text-white hover:bg-white hover:text-green-600">
                <Download className="h-4 w-4 mr-2" />
                Download Proposal
              </Button>
              <Button variant="outline" className="border-white text-white hover:bg-white hover:text-green-600">
                <Mail className="h-4 w-4 mr-2" />
                Contact Sales
              </Button>
            </div>
          </CardContent>
        </Card>

        {/* Next Steps */}
        <Card className="mt-12">
          <CardHeader>
            <CardTitle className="text-3xl text-blue-900">Next Steps</CardTitle>
          </CardHeader>
          <CardContent>
            <p className="text-lg text-gray-700 mb-6">
              Ready to see SportsPro in action? Contact our team to schedule a personalized demonstration 
              and discuss your organization's specific needs.
            </p>
            
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              <div className="bg-gray-50 p-6 rounded-lg border-l-4 border-blue-500">
                <h4 className="font-semibold text-lg mb-3">Schedule Demo</h4>
                <p className="text-sm text-gray-700">Live platform demonstration tailored to your sport and organization size.</p>
              </div>
              <div className="bg-gray-50 p-6 rounded-lg border-l-4 border-blue-500">
                <h4 className="font-semibold text-lg mb-3">Requirements Analysis</h4>
                <p className="text-sm text-gray-700">Detailed assessment of your current processes and integration needs.</p>
              </div>
              <div className="bg-gray-50 p-6 rounded-lg border-l-4 border-blue-500">
                <h4 className="font-semibold text-lg mb-3">Custom Configuration</h4>
                <p className="text-sm text-gray-700">Tailored setup planning to match your specific workflow requirements.</p>
              </div>
              <div className="bg-gray-50 p-6 rounded-lg border-l-4 border-blue-500">
                <h4 className="font-semibold text-lg mb-3">Implementation Support</h4>
                <p className="text-sm text-gray-700">Full guidance and support throughout the deployment process.</p>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Footer */}
        <div className="text-center mt-12 py-8 border-t">
          <p className="text-2xl font-bold text-blue-900 mb-2">SportsPro</p>
          <p className="text-gray-600 italic">Transforming Sports Management Through Technology</p>
          <p className="text-lg font-semibold text-blue-600 mt-2">Professional • Scalable • Secure - Built for Champions</p>
        </div>
      </div>
    </div>
  );
}