import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import PlayerForm from "@/components/forms/player-form";
import { Plus, Edit, Trash2, UserPlus, Search, Filter, Eye, User, Phone, Mail, Calendar, MapPin, Heart, Activity } from "lucide-react";
import type { Player, Team } from "@shared/schema";

export default function Players() {
  const [searchQuery, setSearchQuery] = useState("");
  const [statusFilter, setStatusFilter] = useState<string>("all");
  const [genderFilter, setGenderFilter] = useState<string>("all");
  const [selectedPlayer, setSelectedPlayer] = useState<Player | null>(null);
  const [showCreateDialog, setShowCreateDialog] = useState(false);
  const [showEditDialog, setShowEditDialog] = useState(false);
  const [showViewDialog, setShowViewDialog] = useState(false);
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const { data: players = [], isLoading } = useQuery<(Player & { teamName?: string })[]>({
    queryKey: ["/api/players"],
  });

  const { data: teams = [] } = useQuery<Team[]>({
    queryKey: ["/api/teams"],
  });

  const deletePlayerMutation = useMutation({
    mutationFn: async (playerId: number) => {
      await apiRequest("DELETE", `/api/players/${playerId}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/players"] });
      toast({
        title: "Success",
        description: "Player deleted successfully",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to delete player",
        variant: "destructive",
      });
    },
  });

  const filteredPlayers = players.filter((player: Player & { teamName?: string }) => {
    const matchesSearch = 
      `${player.firstName} ${player.lastName}`.toLowerCase().includes(searchQuery.toLowerCase()) ||
      (player.teamName || "").toLowerCase().includes(searchQuery.toLowerCase()) ||
      (player.position || "").toLowerCase().includes(searchQuery.toLowerCase());
    
    const matchesStatus = statusFilter === "all" || player.status === statusFilter;
    const matchesGender = genderFilter === "all" || player.gender === genderFilter;
    
    return matchesSearch && matchesStatus && matchesGender;
  }) || [];

  const handleView = (player: Player) => {
    setSelectedPlayer(player);
    setShowViewDialog(true);
  };

  const handleEdit = (player: Player) => {
    setSelectedPlayer(player);
    setShowViewDialog(false);
    setShowEditDialog(true);
  };

  const handleDelete = (playerId: number) => {
    if (confirm("Are you sure you want to delete this player?")) {
      deletePlayerMutation.mutate(playerId);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case "active": return "default";
      case "suspended": return "destructive";
      case "inactive": return "secondary";
      default: return "secondary";
    }
  };

  if (isLoading) {
    return (
      <div className="p-4 sm:p-6">
        <div className="animate-pulse space-y-4 sm:space-y-6">
          <div className="h-8 bg-gray-200 rounded w-48"></div>
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 sm:gap-6">
            {[...Array(6)].map((_, i) => (
              <div key={i} className="h-64 bg-gray-200 rounded-xl"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="p-4 sm:p-6">
      {/* Header */}
      <div className="mb-6 sm:mb-8">
        <div className="flex flex-col sm:flex-row sm:items-center justify-between mb-4 gap-4">
          <div>
            <h2 className="text-2xl sm:text-3xl font-bold text-gray-900">Players</h2>
            <p className="text-gray-600">Manage player profiles and information</p>
          </div>
          <Dialog open={showCreateDialog} onOpenChange={setShowCreateDialog}>
            <DialogTrigger asChild>
              <Button className="w-full sm:w-auto">
                <Plus className="h-4 w-4 mr-2" />
                Add Player
              </Button>
            </DialogTrigger>
            <DialogContent className="max-w-4xl max-h-[80vh] overflow-y-auto">
              <DialogHeader>
                <DialogTitle>Add New Player</DialogTitle>
                <DialogDescription>
                  Add a new player to your sports organization
                </DialogDescription>
              </DialogHeader>
              <PlayerForm teams={teams || []} onSuccess={() => setShowCreateDialog(false)} />
            </DialogContent>
          </Dialog>
        </div>

        {/* Filters */}
        <div className="flex flex-col sm:flex-row gap-4">
          <div className="relative flex-1">
            <Search className="absolute left-3 top-3 h-4 w-4 text-gray-400" />
            <Input
              placeholder="Search players..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pl-10"
            />
          </div>
          <Select value={statusFilter} onValueChange={setStatusFilter}>
            <SelectTrigger className="w-full sm:w-40">
              <SelectValue placeholder="Status" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Status</SelectItem>
              <SelectItem value="active">Active</SelectItem>
              <SelectItem value="inactive">Inactive</SelectItem>
              <SelectItem value="suspended">Suspended</SelectItem>
            </SelectContent>
          </Select>
          <Select value={genderFilter} onValueChange={setGenderFilter}>
            <SelectTrigger className="w-full sm:w-40">
              <SelectValue placeholder="Gender" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Genders</SelectItem>
              <SelectItem value="male">Male</SelectItem>
              <SelectItem value="female">Female</SelectItem>
            </SelectContent>
          </Select>
        </div>
      </div>

      {/* Players Grid */}
      {filteredPlayers.length === 0 ? (
        <Card className="text-center py-12">
          <CardContent>
            <UserPlus className="h-12 w-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-semibold text-gray-900 mb-2">No players found</h3>
            <p className="text-gray-500 mb-4">
              {searchQuery || statusFilter !== "all" || genderFilter !== "all" 
                ? "No players match your search criteria" 
                : "Get started by adding your first player"}
            </p>
            {!searchQuery && statusFilter === "all" && genderFilter === "all" && (
              <Button onClick={() => setShowCreateDialog(true)}>
                <Plus className="h-4 w-4 mr-2" />
                Add Player
              </Button>
            )}
          </CardContent>
        </Card>
      ) : (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {filteredPlayers.map((player: Player & { teamName?: string }) => (
            <Card key={player.id} className="hover:shadow-lg transition-all duration-300 hover:scale-[1.02] border-0 shadow-md">
              <CardHeader className="relative overflow-hidden">
                {/* Card Header Background */}
                <div className="absolute inset-0 bg-gradient-to-r from-slate-50 to-blue-50 dark:from-slate-800 dark:to-blue-900"></div>
                
                <div className="relative flex items-center justify-between">
                  <div className="flex items-center space-x-3">
                    {/* Club Logo (Small) */}
                    {teams.find((t: Team) => t.id === player.teamId)?.logoUrl && (
                      <div className="w-8 h-8 bg-white/80 backdrop-blur-sm rounded-full p-1 flex items-center justify-center shadow-sm">
                        <img 
                          src={teams.find((t: Team) => t.id === player.teamId)?.logoUrl || ''} 
                          alt="Club Logo"
                          className="w-6 h-6 object-contain"
                        />
                      </div>
                    )}
                    
                    {/* Player Avatar */}
                    <Avatar className="w-12 h-12 border-2 border-white shadow-sm">
                      <AvatarImage src={player.profileImageUrl ?? undefined} alt={`${player.firstName} ${player.lastName}`} />
                      <AvatarFallback className="bg-primary/10 text-primary font-semibold">
                        {player.firstName[0]}{player.lastName[0]}
                      </AvatarFallback>
                    </Avatar>
                    
                    {/* Player Info */}
                    <div>
                      <CardTitle className="text-lg font-bold">{player.firstName} {player.lastName}</CardTitle>
                      <CardDescription className="flex items-center space-x-2">
                        <span>{player.teamName || "No team assigned"}</span>
                        {player.position && (
                          <>
                            <span className="text-slate-400">•</span>
                            <span className="text-slate-600 dark:text-slate-300">{player.position}</span>
                          </>
                        )}
                      </CardDescription>
                    </div>
                  </div>
                  
                  <div className="flex flex-col items-end space-y-1">
                    <Badge variant={getStatusColor(player.status) as any} className="shadow-sm">
                      {player.status}
                    </Badge>
                    {player.jerseyNumber && (
                      <span className="text-xs font-semibold text-slate-500 bg-white/60 px-2 py-1 rounded-full">
                        #{player.jerseyNumber}
                      </span>
                    )}
                  </div>
                </div>
              </CardHeader>
              <CardContent>
                <div className="space-y-2 mb-4">
                  {player.position && (
                    <p className="text-sm text-gray-600">
                      <span className="font-medium">Position:</span> {player.position}
                    </p>
                  )}
                  {player.jerseyNumber && (
                    <p className="text-sm text-gray-600">
                      <span className="font-medium">Jersey:</span> #{player.jerseyNumber}
                    </p>
                  )}
                  <p className="text-sm text-gray-600">
                    <span className="font-medium">Gender:</span> {player.gender}
                  </p>
                  {player.dateOfBirth && (
                    <p className="text-sm text-gray-600">
                      <span className="font-medium">Age:</span> {
                        new Date().getFullYear() - new Date(player.dateOfBirth).getFullYear()
                      }
                    </p>
                  )}
                </div>
                <div className="flex items-center justify-between">
                  <p className="text-xs text-gray-500">
                    Added {new Date(player.createdAt!).toLocaleDateString()}
                  </p>
                  <div className="flex space-x-1">
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => handleView(player)}
                    >
                      <Eye className="h-4 w-4" />
                    </Button>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => handleEdit(player)}
                    >
                      <Edit className="h-4 w-4" />
                    </Button>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => handleDelete(player.id)}
                      disabled={deletePlayerMutation.isPending}
                    >
                      <Trash2 className="h-4 w-4" />
                    </Button>
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      )}

      {/* View Dialog */}
      <Dialog open={showViewDialog} onOpenChange={setShowViewDialog}>
        <DialogContent className="max-w-5xl w-[98vw] max-h-[95vh] overflow-y-auto p-0">
          <DialogHeader className="sr-only">
            <DialogTitle>Player Details - {selectedPlayer?.firstName} {selectedPlayer?.lastName}</DialogTitle>
            <DialogDescription>
              Complete player information and profile
            </DialogDescription>
          </DialogHeader>
          {selectedPlayer && (
            <div className="space-y-0">
              {/* Professional Gradient Header */}
              <div className="relative overflow-hidden bg-gradient-to-br from-primary via-blue-600 to-purple-600 p-8">
                {/* Decorative Background Elements */}
                <div className="absolute top-0 right-0 w-96 h-96 bg-white/10 rounded-full -translate-y-48 translate-x-48 blur-3xl"></div>
                <div className="absolute bottom-0 left-0 w-72 h-72 bg-white/5 rounded-full translate-y-36 -translate-x-36 blur-2xl"></div>
                
                {/* Content */}
                <div className="relative flex flex-col md:flex-row items-center md:items-start gap-6">
                  {/* Player Avatar */}
                  <div className="relative group">
                    <div className="absolute inset-0 bg-white/20 rounded-2xl blur-xl group-hover:blur-2xl transition-all"></div>
                    <Avatar className="relative w-32 h-32 md:w-40 md:h-40 border-4 border-white/30 shadow-2xl ring-4 ring-white/10">
                      <AvatarImage src={selectedPlayer.profileImageUrl || undefined} alt={`${selectedPlayer.firstName} ${selectedPlayer.lastName}`} />
                      <AvatarFallback className="bg-gradient-to-br from-white/20 to-white/10 text-white text-4xl font-bold">
                        {selectedPlayer.firstName[0]}{selectedPlayer.lastName[0]}
                      </AvatarFallback>
                    </Avatar>
                  </div>

                  {/* Player Info */}
                  <div className="flex-1 text-center md:text-left text-white">
                    <h1 className="text-3xl md:text-4xl font-bold mb-3 drop-shadow-lg">
                      {selectedPlayer.firstName} {selectedPlayer.lastName}
                    </h1>
                    
                    {/* Team & Details */}
                    <p className="text-lg text-white/90 mb-4">
                      {teams?.find(t => t.id === selectedPlayer.teamId)?.name || "No team assigned"}
                    </p>
                    
                    {/* Badges */}
                    <div className="flex flex-wrap items-center justify-center md:justify-start gap-3 mb-4">
                      <Badge className={`px-4 py-1.5 text-sm font-semibold shadow-lg ${
                        selectedPlayer.status === 'active' 
                          ? 'bg-green-500 text-white hover:bg-green-600' 
                          : 'bg-gray-500 text-white hover:bg-gray-600'
                      }`}>
                        {selectedPlayer.status?.toUpperCase() || 'ACTIVE'}
                      </Badge>
                      
                      {selectedPlayer.jerseyNumber && (
                        <div className="flex items-center px-4 py-1.5 bg-white/20 backdrop-blur-sm rounded-full border-2 border-white/30 shadow-lg">
                          <span className="font-bold">#{selectedPlayer.jerseyNumber}</span>
                        </div>
                      )}
                      
                      {selectedPlayer.position && (
                        <div className="px-4 py-1.5 bg-white/20 backdrop-blur-sm rounded-full border-2 border-white/30 shadow-lg">
                          <span className="font-semibold">{selectedPlayer.position}</span>
                        </div>
                      )}
                    </div>

                    {/* Quick Stats */}
                    <div className="grid grid-cols-3 gap-3 mt-6 max-w-md mx-auto md:mx-0">
                      <div className="bg-white/10 backdrop-blur-md rounded-xl p-4 text-center border border-white/20 hover:bg-white/20 transition-all">
                        <div className="text-2xl font-bold">{selectedPlayer.dateOfBirth ? new Date().getFullYear() - new Date(selectedPlayer.dateOfBirth).getFullYear() : 'N/A'}</div>
                        <div className="text-white/80 text-xs mt-1 uppercase tracking-wider">Age</div>
                      </div>
                      <div className="bg-white/10 backdrop-blur-md rounded-xl p-4 text-center border border-white/20 hover:bg-white/20 transition-all">
                        <div className="text-2xl font-bold">{selectedPlayer.height || 'N/A'}</div>
                        <div className="text-white/80 text-xs mt-1 uppercase tracking-wider">Height (cm)</div>
                      </div>
                      <div className="bg-white/10 backdrop-blur-md rounded-xl p-4 text-center border border-white/20 hover:bg-white/20 transition-all">
                        <div className="text-2xl font-bold">{selectedPlayer.weight || 'N/A'}</div>
                        <div className="text-white/80 text-xs mt-1 uppercase tracking-wider">Weight (kg)</div>
                      </div>
                    </div>
                  </div>
                </div>
              </div>

              {/* Information Cards */}
              <div className="p-6 space-y-6">
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-5">
                  {/* Personal Information Card */}
                  <Card className="border-0 shadow-lg hover:shadow-xl transition-all duration-300 bg-gradient-to-br from-purple-50 to-white dark:from-purple-950 dark:to-slate-900">
                    <CardHeader className="pb-3">
                      <CardTitle className="text-lg flex items-center gap-3">
                        <div className="p-2.5 bg-gradient-to-br from-purple-500 to-purple-600 rounded-xl shadow-lg">
                          <User className="h-5 w-5 text-white" />
                        </div>
                        <span className="bg-gradient-to-r from-purple-600 to-purple-800 bg-clip-text text-transparent font-bold">
                          Personal Info
                        </span>
                      </CardTitle>
                    </CardHeader>
                    <CardContent className="space-y-3">
                      <div className="flex justify-between items-center p-3 rounded-xl bg-white/80 dark:bg-slate-800/80 border border-purple-100 dark:border-purple-900 hover:border-purple-300 dark:hover:border-purple-700 transition-colors">
                        <span className="text-sm text-gray-600 dark:text-gray-400 font-semibold">Gender:</span>
                        <span className="text-sm font-bold text-gray-900 dark:text-gray-100 capitalize">{selectedPlayer.gender || 'Not specified'}</span>
                      </div>
                      {selectedPlayer.dateOfBirth && (
                        <div className="flex justify-between items-center p-3 rounded-xl bg-white/80 dark:bg-slate-800/80 border border-purple-100 dark:border-purple-900 hover:border-purple-300 dark:hover:border-purple-700 transition-colors">
                          <span className="text-sm text-gray-600 dark:text-gray-400 font-semibold">Date of Birth:</span>
                          <span className="text-sm font-bold text-gray-900 dark:text-gray-100">{new Date(selectedPlayer.dateOfBirth).toLocaleDateString()}</span>
                        </div>
                      )}
                      {selectedPlayer.nationality && (
                        <div className="flex justify-between items-center p-3 rounded-xl bg-white/80 dark:bg-slate-800/80 border border-purple-100 dark:border-purple-900 hover:border-purple-300 dark:hover:border-purple-700 transition-colors">
                          <span className="text-sm text-gray-600 dark:text-gray-400 font-semibold">Nationality:</span>
                          <span className="text-sm font-bold text-gray-900 dark:text-gray-100">{selectedPlayer.nationality}</span>
                        </div>
                      )}
                    </CardContent>
                  </Card>

                  {/* Contact Information Card */}
                  <Card className="border-0 shadow-lg hover:shadow-xl transition-all duration-300 bg-gradient-to-br from-blue-50 to-white dark:from-blue-950 dark:to-slate-900">
                    <CardHeader className="pb-3">
                      <CardTitle className="text-lg flex items-center gap-3">
                        <div className="p-2.5 bg-gradient-to-br from-blue-500 to-blue-600 rounded-xl shadow-lg">
                          <Mail className="h-5 w-5 text-white" />
                        </div>
                        <span className="bg-gradient-to-r from-blue-600 to-blue-800 bg-clip-text text-transparent font-bold">
                          Contact Info
                        </span>
                      </CardTitle>
                    </CardHeader>
                    <CardContent className="space-y-3">
                      {selectedPlayer.email && (
                        <div className="flex items-center gap-3 p-3 rounded-xl bg-white/80 dark:bg-slate-800/80 border border-blue-100 dark:border-blue-900 hover:border-blue-300 dark:hover:border-blue-700 transition-colors">
                          <Mail className="h-4 w-4 text-blue-600 flex-shrink-0" />
                          <span className="text-sm text-gray-700 dark:text-gray-300 break-all font-medium">{selectedPlayer.email}</span>
                        </div>
                      )}
                      {selectedPlayer.phone && (
                        <div className="flex items-center gap-3 p-3 rounded-xl bg-white/80 dark:bg-slate-800/80 border border-blue-100 dark:border-blue-900 hover:border-blue-300 dark:hover:border-blue-700 transition-colors">
                          <Phone className="h-4 w-4 text-blue-600 flex-shrink-0" />
                          <span className="text-sm text-gray-700 dark:text-gray-300 font-medium">{selectedPlayer.phone}</span>
                        </div>
                      )}
                      {selectedPlayer.address && (
                        <div className="flex items-center gap-3 p-3 rounded-xl bg-white/80 dark:bg-slate-800/80 border border-blue-100 dark:border-blue-900 hover:border-blue-300 dark:hover:border-blue-700 transition-colors">
                          <MapPin className="h-4 w-4 text-blue-600 flex-shrink-0" />
                          <span className="text-sm text-gray-700 dark:text-gray-300 font-medium">{selectedPlayer.address}</span>
                        </div>
                      )}
                      {!selectedPlayer.email && !selectedPlayer.phone && !selectedPlayer.address && (
                        <p className="text-sm text-gray-500 italic p-3">No contact information available</p>
                      )}
                    </CardContent>
                  </Card>

                  {/* Athletic Details Card */}
                  <Card className="border-0 shadow-lg hover:shadow-xl transition-all duration-300 bg-gradient-to-br from-green-50 to-white dark:from-green-950 dark:to-slate-900">
                    <CardHeader className="pb-3">
                      <CardTitle className="text-lg flex items-center gap-3">
                        <div className="p-2.5 bg-gradient-to-br from-green-500 to-green-600 rounded-xl shadow-lg">
                          <Activity className="h-5 w-5 text-white" />
                        </div>
                        <span className="bg-gradient-to-r from-green-600 to-green-800 bg-clip-text text-transparent font-bold">
                          Athletic Stats
                        </span>
                      </CardTitle>
                    </CardHeader>
                    <CardContent className="space-y-3">
                      {selectedPlayer.height && (
                        <div className="flex justify-between items-center p-3 rounded-xl bg-white/80 dark:bg-slate-800/80 border border-green-100 dark:border-green-900 hover:border-green-300 dark:hover:border-green-700 transition-colors">
                          <span className="text-sm text-gray-600 dark:text-gray-400 font-semibold">Height:</span>
                          <span className="text-sm font-bold text-gray-900 dark:text-gray-100">{selectedPlayer.height} cm</span>
                        </div>
                      )}
                      {selectedPlayer.weight && (
                        <div className="flex justify-between items-center p-3 rounded-xl bg-white/80 dark:bg-slate-800/80 border border-green-100 dark:border-green-900 hover:border-green-300 dark:hover:border-green-700 transition-colors">
                          <span className="text-sm text-gray-600 dark:text-gray-400 font-semibold">Weight:</span>
                          <span className="text-sm font-bold text-gray-900 dark:text-gray-100">{selectedPlayer.weight} kg</span>
                        </div>
                      )}
                      {selectedPlayer.preferredFoot && (
                        <div className="flex justify-between items-center p-3 rounded-xl bg-white/80 dark:bg-slate-800/80 border border-green-100 dark:border-green-900 hover:border-green-300 dark:hover:border-green-700 transition-colors">
                          <span className="text-sm text-gray-600 dark:text-gray-400 font-semibold">Preferred Foot:</span>
                          <span className="text-sm font-bold text-gray-900 dark:text-gray-100 capitalize">{selectedPlayer.preferredFoot}</span>
                        </div>
                      )}
                      {selectedPlayer.height && selectedPlayer.weight && (
                        <div className="flex justify-between items-center p-3 rounded-xl bg-white/80 dark:bg-slate-800/80 border border-green-100 dark:border-green-900 hover:border-green-300 dark:hover:border-green-700 transition-colors">
                          <span className="text-sm text-gray-600 dark:text-gray-400 font-semibold">BMI:</span>
                          <span className="text-sm font-bold text-gray-900 dark:text-gray-100">
                            {((selectedPlayer.weight / ((selectedPlayer.height / 100) ** 2))).toFixed(1)}
                          </span>
                        </div>
                      )}
                    </CardContent>
                  </Card>
                </div>

                {/* Health & Medical Card */}
                {(selectedPlayer.emergencyContact || selectedPlayer.medicalNotes || selectedPlayer.bloodType) && (
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-5 mt-5">
                    {selectedPlayer.emergencyContact && (
                      <Card className="border-0 shadow-lg hover:shadow-xl transition-all duration-300 bg-gradient-to-br from-red-50 to-white dark:from-red-950 dark:to-slate-900">
                        <CardHeader className="pb-3">
                          <CardTitle className="text-lg flex items-center gap-3">
                            <div className="p-2.5 bg-gradient-to-br from-red-500 to-red-600 rounded-xl shadow-lg">
                              <Phone className="h-5 w-5 text-white" />
                            </div>
                            <span className="bg-gradient-to-r from-red-600 to-red-800 bg-clip-text text-transparent font-bold">
                              Emergency Contact
                            </span>
                          </CardTitle>
                        </CardHeader>
                        <CardContent>
                          <div className="p-4 bg-white/80 dark:bg-slate-800/80 rounded-xl border border-red-100 dark:border-red-900">
                            <p className="text-sm text-gray-700 dark:text-gray-300 leading-relaxed font-medium">{selectedPlayer.emergencyContact}</p>
                          </div>
                        </CardContent>
                      </Card>
                    )}
                    
                    {selectedPlayer.medicalNotes && (
                      <Card className="border-0 shadow-lg hover:shadow-xl transition-all duration-300 bg-gradient-to-br from-orange-50 to-white dark:from-orange-950 dark:to-slate-900">
                        <CardHeader className="pb-3">
                          <CardTitle className="text-lg flex items-center gap-3">
                            <div className="p-2.5 bg-gradient-to-br from-orange-500 to-orange-600 rounded-xl shadow-lg">
                              <Heart className="h-5 w-5 text-white" />
                            </div>
                            <span className="bg-gradient-to-r from-orange-600 to-orange-800 bg-clip-text text-transparent font-bold">
                              Medical Notes
                            </span>
                          </CardTitle>
                        </CardHeader>
                        <CardContent>
                          <div className="p-4 bg-white/80 dark:bg-slate-800/80 rounded-xl border border-orange-100 dark:border-orange-900">
                            <p className="text-sm text-gray-700 dark:text-gray-300 leading-relaxed font-medium">{selectedPlayer.medicalNotes}</p>
                            {selectedPlayer.bloodType && (
                              <div className="mt-3 pt-3 border-t border-orange-200 dark:border-orange-800">
                                <span className="text-sm font-semibold text-gray-700 dark:text-gray-300">Blood Type: </span>
                                <span className="text-sm font-bold text-orange-600 dark:text-orange-400">{selectedPlayer.bloodType}</span>
                              </div>
                            )}
                          </div>
                        </CardContent>
                      </Card>
                    )}
                  </div>
                )}

                {/* Action Button */}
                <div className="flex justify-end gap-3 pt-4 border-t mt-6">
                  <Button 
                    onClick={() => handleEdit(selectedPlayer)} 
                    className="bg-gradient-to-r from-primary to-blue-600 hover:from-primary/90 hover:to-blue-600/90 shadow-lg"
                    aria-label="Edit player details"
                  >
                    <Edit className="h-4 w-4 mr-2" />
                    Edit Player
                  </Button>
                </div>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>

      {/* Edit Dialog */}
      <Dialog open={showEditDialog} onOpenChange={setShowEditDialog}>
        <DialogContent className="max-w-4xl max-h-[80vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Edit Player</DialogTitle>
            <DialogDescription>
              Update the player information
            </DialogDescription>
          </DialogHeader>
          {selectedPlayer && (
            <PlayerForm
              player={selectedPlayer}
              teams={teams || []}
              onSuccess={() => {
                setShowEditDialog(false);
                setSelectedPlayer(null);
              }}
            />
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}
