import { useState } from "react";
import { useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Progress } from "@/components/ui/progress";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { 
  ArrowLeft, 
  ArrowRight, 
  Check, 
  Star, 
  Users, 
  UserPlus, 
  Target, 
  Trophy, 
  BarChart3,
  Eye,
  BookOpen,
  MousePointer2,
  Video,
  Brain,
  Fingerprint,
  Palette,
  Calendar,
  CreditCard,
  Shield,
  Sparkles,
  Play,
  Zap,
  Camera,
  Settings,
  TrendingUp,
  Award,
  Activity,
  Lock
} from "lucide-react";

import dashboardScreenshot from "@/assets/screenshots/dashboard.svg";
import teamsScreenshot from "@/assets/screenshots/teams.svg";
import playersScreenshot from "@/assets/screenshots/players.svg";
import rosterScreenshot from "@/assets/screenshots/roster.svg";

interface OnboardingStep {
  id: string;
  title: string;
  description: string;
  icon: React.ReactNode;
  content: React.ReactNode;
  actionText?: string;
  actionRoute?: string;
  badge?: string;
}

export default function Onboarding() {
  const [, setLocation] = useLocation();
  const [currentStep, setCurrentStep] = useState(0);
  const [completedSteps, setCompletedSteps] = useState<string[]>([]);

  const steps: OnboardingStep[] = [
    {
      id: "welcome",
      title: "Welcome to SportsPro",
      description: "Your comprehensive AI-powered sports management platform",
      icon: <Star className="h-8 w-8 text-yellow-600" />,
      badge: "New Platform",
      content: (
        <div className="space-y-6">
          <div className="text-center space-y-4">
            <div className="w-20 h-20 bg-gradient-to-br from-primary to-purple-600 rounded-full flex items-center justify-center mx-auto shadow-lg">
              <Trophy className="h-10 w-10 text-white" />
            </div>
            <h3 className="text-2xl font-bold text-gray-900">Transform Your Sports Management</h3>
            <p className="text-gray-600 max-w-md mx-auto">
              Welcome to the most advanced sports management platform with AI-powered features, 
              automated video analysis, and comprehensive team management tools.
            </p>
          </div>
          
          <div className="grid md:grid-cols-2 gap-4">
            <div className="bg-gradient-to-br from-purple-50 to-purple-100 p-4 rounded-xl">
              <Video className="h-6 w-6 text-purple-600 mb-2" />
              <h4 className="font-semibold text-purple-900">AI Video Highlights</h4>
              <p className="text-sm text-purple-700">Automatically generate professional highlight reels</p>
            </div>
            <div className="bg-gradient-to-br from-blue-50 to-blue-100 p-4 rounded-xl">
              <Brain className="h-6 w-6 text-blue-600 mb-2" />
              <h4 className="font-semibold text-blue-900">Smart Onboarding</h4>
              <p className="text-sm text-blue-700">AI-powered setup recommendations</p>
            </div>
            <div className="bg-gradient-to-br from-green-50 to-green-100 p-4 rounded-xl">
              <Fingerprint className="h-6 w-6 text-green-600 mb-2" />
              <h4 className="font-semibold text-green-900">Biometric Security</h4>
              <p className="text-sm text-green-700">Advanced facial recognition & privacy controls</p>
            </div>
            <div className="bg-gradient-to-br from-orange-50 to-orange-100 p-4 rounded-xl">
              <BarChart3 className="h-6 w-6 text-orange-600 mb-2" />
              <h4 className="font-semibold text-orange-900">Real-time Analytics</h4>
              <p className="text-sm text-orange-700">Comprehensive performance tracking</p>
            </div>
          </div>
        </div>
      )
    },
    {
      id: "ai-features",
      title: "AI-Powered Features",
      description: "Discover cutting-edge AI capabilities that set SportsPro apart",
      icon: <Brain className="h-8 w-8 text-blue-600" />,
      badge: "AI-Powered",
      content: (
        <div className="space-y-6">
          <div className="text-center mb-6">
            <div className="w-16 h-16 bg-gradient-to-br from-blue-500 to-purple-600 rounded-full flex items-center justify-center mx-auto mb-4">
              <Sparkles className="h-8 w-8 text-white" />
            </div>
            <h3 className="text-xl font-bold text-gray-900 mb-2">Next-Generation AI Technology</h3>
            <p className="text-gray-600">Advanced artificial intelligence streamlines your sports management</p>
          </div>
          
          <div className="space-y-4">
            <Card className="border-l-4 border-l-purple-500 bg-gradient-to-r from-purple-50 to-white">
              <CardContent className="p-4">
                <div className="flex items-start space-x-3">
                  <Video className="h-6 w-6 text-purple-600 mt-1" />
                  <div>
                    <h4 className="font-semibold text-gray-900">Video Highlight Generator</h4>
                    <p className="text-sm text-gray-600 mt-1">
                      AI automatically detects goals, assists, saves, and key plays from game footage. 
                      Generate professional highlight reels with custom transitions and effects.
                    </p>
                    <div className="flex gap-2 mt-2">
                      <Badge variant="outline" className="text-xs">Auto Detection</Badge>
                      <Badge variant="outline" className="text-xs">HD/4K Export</Badge>
                      <Badge variant="outline" className="text-xs">Custom Effects</Badge>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
            
            <Card className="border-l-4 border-l-blue-500 bg-gradient-to-r from-blue-50 to-white">
              <CardContent className="p-4">
                <div className="flex items-start space-x-3">
                  <Brain className="h-6 w-6 text-blue-600 mt-1" />
                  <div>
                    <h4 className="font-semibold text-gray-900">Smart Onboarding Engine</h4>
                    <p className="text-sm text-gray-600 mt-1">
                      Personalized setup recommendations based on your sport, team size, and objectives. 
                      AI analyzes your needs and guides you through optimal configuration.
                    </p>
                    <div className="flex gap-2 mt-2">
                      <Badge variant="outline" className="text-xs">Sport-Specific</Badge>
                      <Badge variant="outline" className="text-xs">Progress Tracking</Badge>
                      <Badge variant="outline" className="text-xs">AI Analysis</Badge>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
            
            <Card className="border-l-4 border-l-green-500 bg-gradient-to-r from-green-50 to-white">
              <CardContent className="p-4">
                <div className="flex items-start space-x-3">
                  <TrendingUp className="h-6 w-6 text-green-600 mt-1" />
                  <div>
                    <h4 className="font-semibold text-gray-900">Predictive Analytics</h4>
                    <p className="text-sm text-gray-600 mt-1">
                      AI-powered insights predict player performance, injury risks, and team outcomes 
                      based on comprehensive data analysis.
                    </p>
                    <div className="flex gap-2 mt-2">
                      <Badge variant="outline" className="text-xs">Performance Prediction</Badge>
                      <Badge variant="outline" className="text-xs">Risk Assessment</Badge>
                      <Badge variant="outline" className="text-xs">Data Insights</Badge>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      ),
      actionText: "Explore AI Features",
      actionRoute: "/smart-onboarding"
    },
    {
      id: "security-privacy",
      title: "Biometric Security & Privacy",
      description: "Enterprise-grade security with advanced biometric authentication",
      icon: <Fingerprint className="h-8 w-8 text-green-600" />,
      badge: "Enterprise Security",
      content: (
        <div className="space-y-6">
          <div className="text-center mb-6">
            <div className="w-16 h-16 bg-gradient-to-br from-green-500 to-emerald-600 rounded-full flex items-center justify-center mx-auto mb-4">
              <Shield className="h-8 w-8 text-white" />
            </div>
            <h3 className="text-xl font-bold text-gray-900 mb-2">Bank-Level Security</h3>
            <p className="text-gray-600">Advanced biometric authentication and comprehensive privacy controls</p>
          </div>
          
          <div className="grid md:grid-cols-2 gap-4">
            <Card className="bg-gradient-to-br from-green-50 to-emerald-50 border-green-200">
              <CardContent className="p-4">
                <Camera className="h-8 w-8 text-green-600 mb-3" />
                <h4 className="font-semibold text-gray-900 mb-2">Facial Recognition</h4>
                <p className="text-sm text-gray-600 mb-3">
                  Secure biometric authentication using advanced facial recognition technology for player and staff access.
                </p>
                <ul className="text-xs text-gray-600 space-y-1">
                  <li>• Real-time face detection</li>
                  <li>• Liveness verification</li>
                  <li>• Secure data encryption</li>
                </ul>
              </CardContent>
            </Card>
            
            <Card className="bg-gradient-to-br from-blue-50 to-cyan-50 border-blue-200">
              <CardContent className="p-4">
                <Settings className="h-8 w-8 text-blue-600 mb-3" />
                <h4 className="font-semibold text-gray-900 mb-2">Privacy Controls</h4>
                <p className="text-sm text-gray-600 mb-3">
                  Comprehensive privacy settings with three preset levels and granular control over data access.
                </p>
                <ul className="text-xs text-gray-600 space-y-1">
                  <li>• Maximum Privacy preset</li>
                  <li>• Balanced access controls</li>
                  <li>• Open collaboration mode</li>
                </ul>
              </CardContent>
            </Card>
          </div>
          
          <div className="bg-gray-50 p-4 rounded-lg">
            <h4 className="font-semibold text-gray-900 mb-2 flex items-center">
              <Lock className="h-4 w-4 mr-2 text-gray-600" />
              Privacy Features
            </h4>
            <div className="grid md:grid-cols-3 gap-3 text-sm">
              <div className="text-gray-600">
                <strong>Data Collection:</strong> Configurable data gathering with opt-in/opt-out controls
              </div>
              <div className="text-gray-600">
                <strong>Sharing Controls:</strong> Granular permissions for data access and distribution
              </div>
              <div className="text-gray-600">
                <strong>Auto-Deletion:</strong> Scheduled data purging with customizable retention periods
              </div>
            </div>
          </div>
        </div>
      ),
      actionText: "Configure Security",
      actionRoute: "/biometric"
    },
    {
      id: "team-management",
      title: "Advanced Team Management",
      description: "Comprehensive tools for managing players, coaches, and staff",
      icon: <Users className="h-8 w-8 text-orange-600" />,
      content: (
        <div className="space-y-6">
          <div className="text-center mb-6">
            <div className="w-16 h-16 bg-gradient-to-br from-orange-500 to-red-600 rounded-full flex items-center justify-center mx-auto mb-4">
              <Users className="h-8 w-8 text-white" />
            </div>
            <h3 className="text-xl font-bold text-gray-900 mb-2">Complete Team Ecosystem</h3>
            <p className="text-gray-600">Manage every aspect of your sports organization</p>
          </div>
          
          <Tabs defaultValue="players" className="w-full">
            <TabsList className="grid w-full grid-cols-3">
              <TabsTrigger value="players">Players</TabsTrigger>
              <TabsTrigger value="coaches">Coaches</TabsTrigger>
              <TabsTrigger value="analytics">Analytics</TabsTrigger>
            </TabsList>
            
            <TabsContent value="players" className="space-y-4">
              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center space-x-3 mb-3">
                    <UserPlus className="h-6 w-6 text-blue-600" />
                    <h4 className="font-semibold">Player Management</h4>
                  </div>
                  <ul className="space-y-2 text-sm text-gray-600">
                    <li>• Advanced search and filtering by position, age, status</li>
                    <li>• Bulk operations for efficient team management</li>
                    <li>• Player profiles with biometric data integration</li>
                    <li>• Performance tracking and attendance monitoring</li>
                    <li>• Contact management with automated communications</li>
                  </ul>
                </CardContent>
              </Card>
              
              <div className="bg-blue-50 p-4 rounded-lg">
                <img src={playersScreenshot} alt="Player Management" className="w-full rounded-lg shadow-sm" />
              </div>
            </TabsContent>
            
            <TabsContent value="coaches" className="space-y-4">
              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center space-x-3 mb-3">
                    <Award className="h-6 w-6 text-green-600" />
                    <h4 className="font-semibold">Coaching Tools</h4>
                  </div>
                  <ul className="space-y-2 text-sm text-gray-600">
                    <li>• Coach profile management with experience tracking</li>
                    <li>• Team assignment and coaching relationships</li>
                    <li>• Performance feedback and evaluation systems</li>
                    <li>• Training session planning and execution</li>
                    <li>• Communication tools for coach-player interaction</li>
                  </ul>
                </CardContent>
              </Card>
              
              <div className="bg-green-50 p-4 rounded-lg">
                <img src={teamsScreenshot} alt="Team Management" className="w-full rounded-lg shadow-sm" />
              </div>
            </TabsContent>
            
            <TabsContent value="analytics" className="space-y-4">
              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center space-x-3 mb-3">
                    <BarChart3 className="h-6 w-6 text-purple-600" />
                    <h4 className="font-semibold">Performance Analytics</h4>
                  </div>
                  <ul className="space-y-2 text-sm text-gray-600">
                    <li>• Real-time performance dashboards</li>
                    <li>• Player statistics and trend analysis</li>
                    <li>• Team composition and formation optimization</li>
                    <li>• Attendance tracking and engagement metrics</li>
                    <li>• Custom reporting and data visualization</li>
                  </ul>
                </CardContent>
              </Card>
              
              <div className="bg-purple-50 p-4 rounded-lg">
                <img src={dashboardScreenshot} alt="Analytics Dashboard" className="w-full rounded-lg shadow-sm" />
              </div>
            </TabsContent>
          </Tabs>
        </div>
      ),
      actionText: "Manage Teams",
      actionRoute: "/teams"
    },
    {
      id: "customization",
      title: "Platform Customization",
      description: "Personalize your platform with themes, forms, and payment gateways",
      icon: <Palette className="h-8 w-8 text-pink-600" />,
      content: (
        <div className="space-y-6">
          <div className="text-center mb-6">
            <div className="w-16 h-16 bg-gradient-to-br from-pink-500 to-purple-600 rounded-full flex items-center justify-center mx-auto mb-4">
              <Palette className="h-8 w-8 text-white" />
            </div>
            <h3 className="text-xl font-bold text-gray-900 mb-2">Make It Your Own</h3>
            <p className="text-gray-600">Customize every aspect to match your organization's brand</p>
          </div>
          
          <div className="grid md:grid-cols-3 gap-4">
            <Card className="bg-gradient-to-br from-pink-50 to-rose-50 border-pink-200">
              <CardContent className="p-4 text-center">
                <Palette className="h-8 w-8 text-pink-600 mx-auto mb-3" />
                <h4 className="font-semibold text-gray-900 mb-2">18 Theme Designs</h4>
                <p className="text-sm text-gray-600 mb-3">
                  Professional themes from modern gradient to minimalist clean designs
                </p>
                <Badge className="text-xs">Customizable Colors</Badge>
              </CardContent>
            </Card>
            
            <Card className="bg-gradient-to-br from-blue-50 to-cyan-50 border-blue-200">
              <CardContent className="p-4 text-center">
                <BookOpen className="h-8 w-8 text-blue-600 mx-auto mb-3" />
                <h4 className="font-semibold text-gray-900 mb-2">Form Builder</h4>
                <p className="text-sm text-gray-600 mb-3">
                  Drag-and-drop form creation with multi-language support
                </p>
                <Badge className="text-xs">Multi-Language</Badge>
              </CardContent>
            </Card>
            
            <Card className="bg-gradient-to-br from-green-50 to-emerald-50 border-green-200">
              <CardContent className="p-4 text-center">
                <CreditCard className="h-8 w-8 text-green-600 mx-auto mb-3" />
                <h4 className="font-semibold text-gray-900 mb-2">Payment Gateways</h4>
                <p className="text-sm text-gray-600 mb-3">
                  Support for 9 major payment providers worldwide
                </p>
                <Badge className="text-xs">Global Support</Badge>
              </CardContent>
            </Card>
          </div>
          
          <div className="space-y-3">
            <h4 className="font-semibold text-gray-900">Available Customizations:</h4>
            <div className="grid md:grid-cols-2 gap-3 text-sm">
              <div className="flex items-center space-x-2">
                <Check className="h-4 w-4 text-green-500" />
                <span>Custom color schemes and gradients</span>
              </div>
              <div className="flex items-center space-x-2">
                <Check className="h-4 w-4 text-green-500" />
                <span>Logo and branding integration</span>
              </div>
              <div className="flex items-center space-x-2">
                <Check className="h-4 w-4 text-green-500" />
                <span>Multi-language interface support</span>
              </div>
              <div className="flex items-center space-x-2">
                <Check className="h-4 w-4 text-green-500" />
                <span>Custom form fields and layouts</span>
              </div>
              <div className="flex items-center space-x-2">
                <Check className="h-4 w-4 text-green-500" />
                <span>Payment gateway configuration</span>
              </div>
              <div className="flex items-center space-x-2">
                <Check className="h-4 w-4 text-green-500" />
                <span>Responsive design optimization</span>
              </div>
            </div>
          </div>
        </div>
      ),
      actionText: "Customize Platform",
      actionRoute: "/theme-settings"
    },
    {
      id: "get-started",
      title: "You're All Set!",
      description: "Start managing your team with SportsPro's powerful features",
      icon: <Trophy className="h-8 w-8 text-yellow-600" />,
      content: (
        <div className="space-y-6 text-center">
          <div className="space-y-4">
            <div className="w-20 h-20 bg-gradient-to-br from-yellow-400 to-orange-500 rounded-full flex items-center justify-center mx-auto shadow-lg">
              <Trophy className="h-10 w-10 text-white" />
            </div>
            <h3 className="text-2xl font-bold text-gray-900">Ready to Transform Your Team!</h3>
            <p className="text-gray-600 max-w-md mx-auto">
              You're now ready to leverage all of SportsPro's advanced features. 
              Start with your dashboard to see the full power of AI-driven sports management.
            </p>
          </div>
          
          <div className="grid md:grid-cols-2 gap-4">
            <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
              <CardContent className="p-4">
                <BarChart3 className="h-6 w-6 text-blue-600 mb-2" />
                <h4 className="font-semibold text-blue-900">Dashboard</h4>
                <p className="text-sm text-blue-700">View real-time analytics and team overview</p>
              </CardContent>
            </Card>
            
            <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
              <CardContent className="p-4">
                <Users className="h-6 w-6 text-green-600 mb-2" />
                <h4 className="font-semibold text-green-900">Team Management</h4>
                <p className="text-sm text-green-700">Add players and organize your roster</p>
              </CardContent>
            </Card>
            
            <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
              <CardContent className="p-4">
                <Video className="h-6 w-6 text-purple-600 mb-2" />
                <h4 className="font-semibold text-purple-900">Video Highlights</h4>
                <p className="text-sm text-purple-700">Upload footage for AI analysis</p>
              </CardContent>
            </Card>
            
            <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
              <CardContent className="p-4">
                <Brain className="h-6 w-6 text-orange-600 mb-2" />
                <h4 className="font-semibold text-orange-900">Smart Onboarding</h4>
                <p className="text-sm text-orange-700">Get AI-powered setup recommendations</p>
              </CardContent>
            </Card>
          </div>
          
          <div className="flex flex-col sm:flex-row gap-3 justify-center">
            <Button 
              size="lg" 
              onClick={() => setLocation("/dashboard")}
              className="bg-gradient-to-r from-primary to-purple-600 hover:from-primary/90 hover:to-purple-600/90"
            >
              <BarChart3 className="mr-2 h-5 w-5" />
              Go to Dashboard
            </Button>
            <Button 
              size="lg" 
              variant="outline"
              onClick={() => setLocation("/smart-onboarding")}
            >
              <Brain className="mr-2 h-5 w-5" />
              Smart Setup
            </Button>
          </div>
        </div>
      ),
      actionText: "Go to Dashboard",
      actionRoute: "/dashboard"
    }
  ];

  const nextStep = () => {
    if (currentStep < steps.length - 1) {
      setCurrentStep(currentStep + 1);
      if (!completedSteps.includes(steps[currentStep].id)) {
        setCompletedSteps([...completedSteps, steps[currentStep].id]);
      }
    }
  };

  const prevStep = () => {
    if (currentStep > 0) {
      setCurrentStep(currentStep - 1);
    }
  };

  const goToStep = (stepIndex: number) => {
    setCurrentStep(stepIndex);
  };

  const progress = ((currentStep + 1) / steps.length) * 100;

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-white to-slate-50">
      {/* Header */}
      <header className="border-b bg-white/80 backdrop-blur-lg">
        <div className="container mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-3">
              <div className="w-10 h-10 bg-gradient-to-br from-primary to-purple-600 rounded-lg flex items-center justify-center">
                <Trophy className="h-6 w-6 text-white" />
              </div>
              <div>
                <h1 className="text-xl font-bold text-gray-900">SportsPro</h1>
                <p className="text-xs text-gray-500">Platform Tour</p>
              </div>
            </div>
            
            <Button 
              variant="outline" 
              onClick={() => setLocation("/dashboard")}
              className="text-sm"
            >
              Skip Tour
            </Button>
          </div>
        </div>
      </header>

      <div className="container mx-auto px-4 py-8">
        <div className="max-w-4xl mx-auto">
          {/* Progress Bar */}
          <div className="mb-8">
            <div className="flex items-center justify-between mb-4">
              <h2 className="text-2xl font-bold text-gray-900">Platform Tour</h2>
              <span className="text-sm text-gray-500">
                Step {currentStep + 1} of {steps.length}
              </span>
            </div>
            <Progress value={progress} className="h-2" />
          </div>

          {/* Step Navigation */}
          <div className="flex space-x-2 mb-8 overflow-x-auto pb-2">
            {steps.map((step, index) => (
              <button
                key={step.id}
                onClick={() => goToStep(index)}
                className={`flex items-center space-x-2 px-4 py-2 rounded-lg text-sm font-medium transition-all whitespace-nowrap ${
                  index === currentStep
                    ? 'bg-primary text-white shadow-lg'
                    : index < currentStep
                    ? 'bg-green-100 text-green-700 hover:bg-green-200'
                    : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                }`}
              >
                {index < currentStep ? (
                  <Check className="h-4 w-4" />
                ) : (
                  <span className="w-4 h-4 rounded-full border-2 border-current flex items-center justify-center text-xs">
                    {index + 1}
                  </span>
                )}
                <span>{step.title}</span>
                {step.badge && (
                  <Badge variant="secondary" className="text-xs">
                    {step.badge}
                  </Badge>
                )}
              </button>
            ))}
          </div>

          {/* Main Content */}
          <Card className="mb-8 shadow-xl">
            <CardHeader className="text-center bg-gradient-to-r from-primary/5 to-purple-500/5">
              <div className="flex items-center justify-center space-x-3 mb-4">
                <div className="w-12 h-12 bg-gradient-to-br from-primary to-purple-600 rounded-xl flex items-center justify-center">
                  {steps[currentStep].icon}
                </div>
                {steps[currentStep].badge && (
                  <Badge className="bg-gradient-to-r from-primary to-purple-600 text-white">
                    {steps[currentStep].badge}
                  </Badge>
                )}
              </div>
              <CardTitle className="text-2xl text-gray-900">
                {steps[currentStep].title}
              </CardTitle>
              <CardDescription className="text-lg">
                {steps[currentStep].description}
              </CardDescription>
            </CardHeader>
            <CardContent className="p-8">
              {steps[currentStep].content}
            </CardContent>
          </Card>

          {/* Navigation */}
          <div className="flex items-center justify-between">
            <Button
              variant="outline"
              onClick={prevStep}
              disabled={currentStep === 0}
              className="flex items-center space-x-2"
            >
              <ArrowLeft className="h-4 w-4" />
              <span>Previous</span>
            </Button>

            <div className="flex space-x-3">
              {steps[currentStep].actionRoute && (
                <Button
                  variant="outline"
                  onClick={() => setLocation(steps[currentStep].actionRoute!)}
                  className="flex items-center space-x-2"
                >
                  <Eye className="h-4 w-4" />
                  <span>{steps[currentStep].actionText || "Explore"}</span>
                </Button>
              )}

              {currentStep < steps.length - 1 ? (
                <Button
                  onClick={nextStep}
                  className="flex items-center space-x-2 bg-gradient-to-r from-primary to-purple-600 hover:from-primary/90 hover:to-purple-600/90"
                >
                  <span>Next</span>
                  <ArrowRight className="h-4 w-4" />
                </Button>
              ) : (
                <Button
                  onClick={() => setLocation("/dashboard")}
                  className="flex items-center space-x-2 bg-gradient-to-r from-green-500 to-emerald-600 hover:from-green-600 hover:to-emerald-700"
                >
                  <span>Complete Tour</span>
                  <Check className="h-4 w-4" />
                </Button>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}