import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { 
  Trophy, Users, BarChart3, Shield, Menu, X, Target, MessageSquare, 
  CreditCard, Globe, Star, CheckCircle, PlayCircle, UserPlus, Calendar, 
  Award, Zap, TrendingUp, Clock, MapPin, Video, Brain, Fingerprint,
  Palette, Camera, ArrowRight, ChevronRight, Sparkles, Rocket,
  Database, Lock, Cloud, Smartphone, Monitor, Headphones, Download,
  FileText, Settings
} from "lucide-react";
import { useState, useEffect } from "react";
import { useQuery } from "@tanstack/react-query";
import dashboardScreenshot from "@/assets/screenshots/dashboard.svg";
import teamsScreenshot from "@/assets/screenshots/teams.svg";
import playersScreenshot from "@/assets/screenshots/players.svg";
import rosterScreenshot from "@/assets/screenshots/roster.svg";
import SEOHead, { seoConfigs } from "@/components/SEOHead";

// Icon mapping for dynamic features from CMS
const iconMap: Record<string, JSX.Element> = {
  Video: <Video className="h-8 w-8" />,
  Brain: <Brain className="h-8 w-8" />,
  Fingerprint: <Fingerprint className="h-8 w-8" />,
  Users: <Users className="h-8 w-8" />,
  BarChart3: <BarChart3 className="h-8 w-8" />,
  Trophy: <Trophy className="h-8 w-8" />,
  Shield: <Shield className="h-8 w-8" />,
  UserPlus: <UserPlus className="h-8 w-8" />,
  Globe: <Globe className="h-8 w-8" />,
  CheckCircle: <CheckCircle className="h-8 w-8" />,
};

export default function Landing() {
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const [activeFeature, setActiveFeature] = useState(0);

  // Quick Actions handlers
  const handleDownloadApplication = () => {
    // Create a comprehensive application package as a zip file
    const appData = {
      name: "SportsPro-Setup-Package",
      version: "1.0.0",
      description: "Complete SportsPro sports management platform setup package",
      installation: {
        requirements: {
          node: ">=20.0.0",
          database: "PostgreSQL 15+",
          memory: "4GB RAM minimum",
          storage: "10GB available space"
        },
        steps: [
          "1. Install Node.js 20+ and PostgreSQL",
          "2. Clone or extract the SportsPro application",
          "3. Run 'npm install' to install dependencies",
          "4. Configure environment variables (.env file)",
          "5. Run database migrations: npm run db:push",
          "6. Start the application: npm run dev",
          "7. Access at http://localhost:5000"
        ]
      },
      features: [
        "Multi-sport team management",
        "Player performance tracking",
        "AI-powered video highlights",
        "Biometric data integration",
        "Attendance tracking",
        "Email notifications",
        "Payment processing",
        "CMS for content management"
      ],
      support: {
        documentation: "https://docs.sportspro.com",
        community: "https://community.sportspro.com",
        email: "support@sportspro.com"
      }
    };

    const blob = new Blob([JSON.stringify(appData, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'SportsPro-Setup-Package.json';
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  };

  const handleDownloadTemplate = () => {
    // Create environment template
    const envTemplate = `# SportsPro Environment Configuration Template
# Copy this file to .env and fill in your actual values

# Database Configuration
DATABASE_URL=postgresql://username:password@localhost:5432/sportspro
PGHOST=localhost
PGPORT=5432
PGUSER=your_username
PGPASSWORD=your_password
PGDATABASE=sportspro

# Application Configuration
NODE_ENV=development
SESSION_SECRET=your-super-secret-session-key-here-change-this
PORT=5000

# Email Configuration (SendGrid)
SENDGRID_API_KEY=your_sendgrid_api_key_here
FROM_EMAIL=noreply@yourdomain.com

# Authentication (if using Replit Auth)
REPL_ID=your_repl_id
ISSUER_URL=https://replit.com/oidc
REPLIT_DOMAINS=your-repl-domain.replit.dev

# Payment Gateways (Optional - configure as needed)
STRIPE_SECRET_KEY=sk_test_your_stripe_secret_key
VITE_STRIPE_PUBLIC_KEY=pk_test_your_stripe_public_key

# PayPal (Optional)
PAYPAL_CLIENT_ID=your_paypal_client_id
PAYPAL_CLIENT_SECRET=your_paypal_client_secret

# AI Services (Optional)
OPENAI_API_KEY=your_openai_api_key
ANTHROPIC_API_KEY=your_anthropic_api_key
GEMINI_API_KEY=your_gemini_api_key

# File Upload Configuration
MAX_FILE_SIZE=10485760
UPLOAD_PATH=./uploads

# Security Configuration
CORS_ORIGIN=http://localhost:5000
RATE_LIMIT_WINDOW=15
RATE_LIMIT_MAX_REQUESTS=100

# Logging Configuration
LOG_LEVEL=info
LOG_TO_FILE=true

# Application Features
ENABLE_VIDEO_HIGHLIGHTS=true
ENABLE_BIOMETRIC_TRACKING=true
ENABLE_EMAIL_NOTIFICATIONS=true
ENABLE_PAYMENT_PROCESSING=true

# Social Media Integration (Optional)
FACEBOOK_APP_ID=your_facebook_app_id
TWITTER_API_KEY=your_twitter_api_key
INSTAGRAM_API_KEY=your_instagram_api_key

# Backup Configuration
BACKUP_ENABLED=true
BACKUP_SCHEDULE=0 2 * * *
BACKUP_RETENTION_DAYS=30
`;

    const blob = new Blob([envTemplate], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = '.env.template';
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  };

  const handleDownloadConfigs = () => {
    // Create configuration examples
    const configExamples = {
      "football-team": {
        name: "Professional Football Team Configuration",
        sport: "football",
        teamSize: 25,
        positions: [
          "Goalkeeper", "Right Back", "Left Back", "Centre Back", "Defensive Midfielder",
          "Central Midfielder", "Attacking Midfielder", "Right Winger", "Left Winger",
          "Striker", "Centre Forward"
        ],
        formation: "4-3-3",
        trainingSchedule: {
          monday: "Technical Skills - 2 hours",
          tuesday: "Physical Conditioning - 1.5 hours",
          wednesday: "Tactical Training - 2 hours",
          thursday: "Recovery/Light Training - 1 hour",
          friday: "Match Preparation - 1.5 hours",
          saturday: "Match Day",
          sunday: "Rest Day"
        },
        performanceMetrics: [
          "Goals", "Assists", "Pass Accuracy", "Distance Covered", 
          "Sprint Speed", "Heart Rate", "Recovery Time"
        ]
      },
      "basketball-team": {
        name: "Professional Basketball Team Configuration",
        sport: "basketball",
        teamSize: 15,
        positions: [
          "Point Guard", "Shooting Guard", "Small Forward", 
          "Power Forward", "Center"
        ],
        formation: "5-player rotation",
        trainingSchedule: {
          monday: "Shooting Practice - 2 hours",
          tuesday: "Conditioning & Strength - 1.5 hours",
          wednesday: "Team Tactics - 2 hours",
          thursday: "Individual Skills - 1.5 hours",
          friday: "Game Preparation - 1 hour",
          saturday: "Game Day",
          sunday: "Recovery/Film Study - 1 hour"
        },
        performanceMetrics: [
          "Points", "Rebounds", "Assists", "Field Goal %", 
          "3-Point %", "Free Throw %", "Vertical Jump", "Sprint Time"
        ]
      },
      "youth-soccer": {
        name: "Youth Soccer League Configuration",
        sport: "soccer",
        teamSize: 16,
        ageGroup: "U-16",
        positions: [
          "Goalkeeper", "Defender", "Midfielder", "Forward"
        ],
        formation: "4-4-2",
        trainingSchedule: {
          tuesday: "Skills Development - 1.5 hours",
          thursday: "Team Play & Tactics - 1.5 hours",
          saturday: "Match Day",
          sunday: "Fun Games & Conditioning - 1 hour"
        },
        performanceMetrics: [
          "Participation", "Improvement", "Teamwork", "Sportsmanship",
          "Basic Skills", "Fitness Level"
        ],
        parentCommunication: {
          weeklyNewsletter: true,
          gameUpdates: true,
          practiceReminders: true
        }
      },
      "multi-sport-club": {
        name: "Multi-Sport Club Configuration",
        sports: ["tennis", "swimming", "track"],
        totalMembers: 150,
        facilities: [
          "Tennis Courts", "Swimming Pool", "Track Field", 
          "Gym", "Locker Rooms", "Medical Room"
        ],
        programs: {
          tennis: {
            levels: ["Beginner", "Intermediate", "Advanced", "Competitive"],
            sessionDuration: "1 hour",
            maxPlayersPerSession: 8
          },
          swimming: {
            levels: ["Learn to Swim", "Stroke Development", "Competitive"],
            sessionDuration: "45 minutes",
            maxPlayersPerSession: 12
          },
          track: {
            events: ["Sprints", "Middle Distance", "Long Distance", "Field Events"],
            sessionDuration: "1.5 hours",
            maxPlayersPerSession: 20
          }
        },
        membershipTiers: [
          { name: "Basic", sports: 1, price: "$50/month" },
          { name: "Premium", sports: 2, price: "$80/month" },
          { name: "Elite", sports: "All", price: "$120/month" }
        ]
      }
    };

    const blob = new Blob([JSON.stringify(configExamples, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'SportsPro-Configuration-Examples.json';
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  };

  // Fetch CMS content for landing page
  const { data: cmsData, isLoading: cmsLoading } = useQuery({
    queryKey: ["/api/cms/landing"],
    retry: false,
  });

  const features = [
    {
      icon: <Video className="h-8 w-8" />,
      title: "AI Video Highlights",
      description: "Automatically generate professional highlight reels with AI-powered analysis",
      color: "from-purple-500 to-pink-500"
    },
    {
      icon: <Brain className="h-8 w-8" />,
      title: "Smart Onboarding",
      description: "Intelligent setup recommendations based on your sport and team structure",
      color: "from-blue-500 to-cyan-500"
    },
    {
      icon: <Fingerprint className="h-8 w-8" />,
      title: "Biometric Security",
      description: "Advanced facial recognition and privacy controls for player data",
      color: "from-green-500 to-emerald-500"
    },
    {
      icon: <BarChart3 className="h-8 w-8" />,
      title: "Performance Analytics",
      description: "Comprehensive statistics and real-time performance tracking",
      color: "from-orange-500 to-red-500"
    }
  ];

  useEffect(() => {
    const interval = setInterval(() => {
      setActiveFeature((prev) => (prev + 1) % features.length);
    }, 4000);
    return () => clearInterval(interval);
  }, []);

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-white to-slate-50">
      <SEOHead {...seoConfigs.landing} />
      
      {/* Header */}
      <header className="sticky top-0 z-50 border-b bg-white/80 backdrop-blur-lg shadow-sm">
        <div className="container mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-3">
              <div className="w-12 h-12 bg-gradient-to-br from-primary to-purple-600 rounded-xl flex items-center justify-center shadow-lg">
                <Trophy className="h-7 w-7 text-white" />
              </div>
              <div>
                <h1 className="text-2xl font-bold bg-gradient-to-r from-gray-900 to-gray-700 bg-clip-text text-transparent">
                  SportsPro
                </h1>
                <p className="text-xs text-gray-500 font-medium">Professional Sports Management</p>
              </div>
            </div>
            
            {/* Desktop Navigation */}
            <nav className="hidden lg:flex items-center space-x-8">
              <a href="/" className="text-primary font-semibold">Home</a>
              <a href="/about" className="text-gray-600 hover:text-primary transition-colors font-medium">About</a>
              <a href="/services" className="text-gray-600 hover:text-primary transition-colors font-medium">Features</a>
              <a href="/contact" className="text-gray-600 hover:text-primary transition-colors font-medium">Contact</a>
              <div className="h-6 w-px bg-gray-200"></div>
              <Button variant="outline" className="font-medium">Sign In</Button>
            </nav>
            
            <div className="flex items-center space-x-3">
              <Button 
                onClick={() => window.location.href = "/auth"} 
                size="lg" 
                className="hidden sm:inline-flex bg-gradient-to-r from-primary to-purple-600 hover:from-primary/90 hover:to-purple-600/90 font-semibold shadow-lg"
              >
                Get Started Free
                <ArrowRight className="ml-2 h-4 w-4" />
              </Button>
              
              {/* Mobile menu button */}
              <Button
                variant="outline"
                size="sm"
                className="lg:hidden"
                onClick={() => setMobileMenuOpen(!mobileMenuOpen)}
              >
                {mobileMenuOpen ? <X className="h-4 w-4" /> : <Menu className="h-4 w-4" />}
              </Button>
            </div>
          </div>
          
          {/* Mobile Navigation */}
          {mobileMenuOpen && (
            <div className="lg:hidden mt-4 pt-4 border-t bg-white rounded-lg shadow-lg">
              <nav className="flex flex-col space-y-3 p-4">
                <a href="/" className="text-primary font-semibold py-2">Home</a>
                <a href="/about" className="text-gray-600 hover:text-primary transition-colors py-2">About</a>
                <a href="/services" className="text-gray-600 hover:text-primary transition-colors py-2">Features</a>
                <a href="/contact" className="text-gray-600 hover:text-primary transition-colors py-2">Contact</a>
                <div className="pt-2">
                  <Button onClick={() => window.location.href = "/auth"} className="w-full">
                    Get Started Free
                  </Button>
                </div>
              </nav>
            </div>
          )}
        </div>
      </header>

      {/* Hero Section */}
      <section className="relative py-20 lg:py-32 overflow-hidden">
        <div className="absolute inset-0 bg-gradient-to-br from-primary/5 via-purple-500/5 to-pink-500/5"></div>
        <div className="relative container mx-auto px-4">
          <div className="grid lg:grid-cols-2 gap-12 items-center">
            <div className="space-y-8">
              <div className="space-y-4">
                {cmsData && (cmsData as any)?.contentBlocks?.find((block: any) => block.blockKey === 'hero') ? (
                  <>
                    <Badge className="bg-gradient-to-r from-primary to-purple-600 text-white border-0 shadow-lg">
                      <Sparkles className="h-3 w-3 mr-1" />
                      {(cmsData as any).contentBlocks.find((block: any) => block.blockKey === 'hero')?.metadata?.badge || 'New: AI-Powered Video Highlights'}
                    </Badge>
                    <h1 className="text-3xl sm:text-4xl md:text-5xl lg:text-6xl font-bold leading-tight">
                      <span className="bg-gradient-to-r from-gray-900 via-primary to-purple-600 bg-clip-text text-transparent">
                        {(cmsData as any).contentBlocks.find((block: any) => block.blockKey === 'hero')?.title || 'Professional Sports Management'}
                      </span>
                    </h1>
                    <p className="text-lg sm:text-xl text-gray-600 leading-relaxed max-w-xl">
                      {(cmsData as any).contentBlocks.find((block: any) => block.blockKey === 'hero')?.subtitle || 'Transform your sports organization with AI-powered analytics'}
                    </p>
                  </>
                ) : (
                  <>
                    <Badge className="bg-gradient-to-r from-primary to-purple-600 text-white border-0 shadow-lg">
                      <Sparkles className="h-3 w-3 mr-1" />
                      New: AI-Powered Video Highlights
                    </Badge>
                    <h1 className="text-3xl sm:text-4xl md:text-5xl lg:text-6xl font-bold leading-tight">
                      <span className="bg-gradient-to-r from-gray-900 via-primary to-purple-600 bg-clip-text text-transparent">
                        Professional
                      </span>
                      <br />
                      <span className="text-gray-900">Sports Management</span>
                    </h1>
                    <p className="text-lg sm:text-xl text-gray-600 leading-relaxed max-w-xl">
                      Transform your sports organization with AI-powered analytics, automated video highlights, 
                      and comprehensive team management tools trusted by professional clubs worldwide.
                    </p>
                  </>
                )}
              </div>
              
              <div className="flex flex-col sm:flex-row gap-4">
                <Button 
                  size="lg" 
                  onClick={() => window.location.href = (cmsData as any)?.contentBlocks?.find((block: any) => block.blockKey === 'hero')?.buttonUrl || "/auth"}
                  className="bg-gradient-to-r from-primary to-purple-600 hover:from-primary/90 hover:to-purple-600/90 text-lg px-8 py-4 shadow-xl"
                >
                  {(cmsData as any)?.contentBlocks?.find((block: any) => block.blockKey === 'hero')?.buttonText || "Start Free Trial"}
                  <Rocket className="ml-2 h-5 w-5" />
                </Button>
                <Button 
                  size="lg" 
                  variant="outline" 
                  className="text-lg px-8 py-4 border-2 border-primary text-primary hover:bg-primary hover:text-white transition-all font-semibold"
                >
                  <PlayCircle className="mr-2 h-5 w-5" />
                  Watch Demo
                </Button>
              </div>
              
              <div className="flex flex-col sm:flex-row sm:items-center space-y-2 sm:space-y-0 sm:space-x-6 text-sm text-gray-600">
                <div className="flex items-center">
                  <CheckCircle className="h-4 w-4 text-green-500 mr-2" />
                  No Credit Card Required
                </div>
                <div className="flex items-center">
                  <CheckCircle className="h-4 w-4 text-green-500 mr-2" />
                  14-Day Free Trial
                </div>
              </div>
            </div>
            
            {/* Dynamic Feature Showcase */}
            <div className="relative mt-12 lg:mt-0">
              <div className="relative bg-gradient-to-br from-white to-gray-50 rounded-3xl shadow-2xl overflow-hidden">
                <div className="p-4 sm:p-6 lg:p-8">
                  <div className={`bg-gradient-to-r ${features[activeFeature].color} rounded-2xl p-4 sm:p-6 text-white mb-4 sm:mb-6`}>
                    <div className="flex flex-col sm:flex-row sm:items-center space-y-2 sm:space-y-0 sm:space-x-4 mb-4">
                      <div className="w-8 h-8 sm:w-10 sm:h-10 flex items-center justify-center">
                        {features[activeFeature].icon}
                      </div>
                      <h3 className="text-lg sm:text-xl font-bold">{features[activeFeature].title}</h3>
                    </div>
                    <p className="text-white/90 text-sm sm:text-base">{features[activeFeature].description}</p>
                  </div>
                  
                  <div className="grid grid-cols-1 sm:grid-cols-2 gap-3">
                    {features.map((feature, index) => (
                      <div
                        key={index}
                        className={`p-3 rounded-lg cursor-pointer transition-all ${
                          index === activeFeature 
                            ? 'bg-primary/10 border-2 border-primary' 
                            : 'bg-gray-50 hover:bg-gray-100 border-2 border-transparent'
                        }`}
                        onClick={() => setActiveFeature(index)}
                      >
                        <div className="flex items-center space-x-2">
                          <div className={`w-6 h-6 sm:w-8 sm:h-8 rounded-lg flex items-center justify-center ${
                            index === activeFeature ? 'bg-primary text-white' : 'bg-gray-200 text-gray-600'
                          }`}>
                            {feature.icon}
                          </div>
                          <span className="text-xs sm:text-sm font-medium text-gray-900">{feature.title}</span>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Features Grid */}
      <section className="py-20 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="text-center mb-16">
            <h2 className="text-4xl font-bold text-gray-900 mb-4">
              Everything You Need to Manage Your Team
            </h2>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">
              From AI-powered video analysis to comprehensive player management, 
              SportsPro provides all the tools professional sports organizations need.
            </p>
          </div>
          
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6 sm:gap-8">
            {((cmsData as any)?.features && (cmsData as any).features.length > 0 ? (cmsData as any).features : [
              {
                icon: "Video",
                title: "AI Video Highlights",
                description: "Automatically detect and compile highlight reels from game footage with advanced AI analysis.",
                features: ["Auto highlight detection", "Multiple export formats", "Custom transitions"],
                color: "from-purple-500 to-pink-500"
              },
              {
                icon: "Brain",
                title: "Smart Onboarding",
                description: "Get personalized setup recommendations based on your sport, team size, and objectives.",
                features: ["Sport-specific guidance", "Setup scoring", "Progress tracking"],
                color: "from-blue-500 to-cyan-500"
              },
              {
                icon: "Fingerprint",
                title: "Biometric Security",
                description: "Advanced facial recognition and comprehensive privacy controls for sensitive player data.",
                features: ["Facial recognition", "Privacy presets", "Secure data handling"],
                color: "from-green-500 to-emerald-500"
              },
              {
                icon: "Users",
                title: "Team Management",
                description: "Complete player roster management with advanced search, filtering, and bulk operations.",
                features: ["Player profiles", "Team organization", "Bulk actions"],
                color: "from-orange-500 to-red-500"
              }
            ]).map((feature: any, index: number) => (
              <Card key={index} className="relative group hover:shadow-xl transition-all duration-300 border-0 shadow-lg">
                <div className="absolute inset-0 bg-gradient-to-br from-white to-gray-50 rounded-lg"></div>
                <CardHeader className="relative p-4 sm:p-6">
                  <div className="w-10 h-10 sm:w-12 sm:h-12 bg-gray-100 rounded-xl flex items-center justify-center mb-4 group-hover:scale-110 transition-transform">
                    {iconMap[feature.icon] || <BarChart3 className="h-8 w-8" />}
                  </div>
                  <CardTitle className="text-lg sm:text-xl font-bold text-gray-900">{feature.title}</CardTitle>
                  <CardDescription className="text-sm sm:text-base text-gray-600">{feature.description}</CardDescription>
                </CardHeader>
                <CardContent className="relative p-4 sm:p-6 pt-0">
                  <ul className="space-y-2">
                    {(Array.isArray(feature.features) ? feature.features : []).map((item: any, idx: any) => (
                      <li key={idx} className="flex items-center text-xs sm:text-sm text-gray-600">
                        <CheckCircle className="h-3 w-3 sm:h-4 sm:w-4 text-green-500 mr-2 flex-shrink-0" />
                        {item}
                      </li>
                    ))}
                  </ul>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Social Proof */}
      <section className="py-20">
        <div className="container mx-auto px-4 text-center">
          <h2 className="text-3xl font-bold text-gray-900 mb-4">
            Trusted by Sports Organizations Worldwide
          </h2>
          <p className="text-gray-600 mb-12">
            Join thousands of teams already using SportsPro to manage their operations
          </p>
          
          <div className="grid grid-cols-2 sm:grid-cols-4 gap-4 sm:gap-8 items-center opacity-60">
            {[
              { name: "Premier League", logo: "🏆" },
              { name: "NBA Teams", logo: "🏀" },
              { name: "UEFA Clubs", logo: "⚽" },
              { name: "Olympic Teams", logo: "🥇" }
            ].map((org, index) => (
              <div key={index} className="flex flex-col items-center space-y-2">
                <div className="text-3xl sm:text-4xl">{org.logo}</div>
                <span className="text-xs sm:text-sm font-medium text-gray-500 text-center">{org.name}</span>
              </div>
            ))}
          </div>
          
          <div className="mt-12 sm:mt-16 grid grid-cols-1 sm:grid-cols-3 gap-6 sm:gap-8">
            {[
              { stat: "10,000+", label: "Active Teams" },
              { stat: "500,000+", label: "Players Managed" },
              { stat: "99.9%", label: "Uptime" }
            ].map((item, index) => (
              <div key={index} className="text-center">
                <div className="text-3xl sm:text-4xl font-bold text-primary mb-2">{item.stat}</div>
                <div className="text-sm sm:text-base text-gray-600">{item.label}</div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Quick Actions Section */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">Quick Actions</h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              Get started instantly with our ready-to-use resources and tools
            </p>
          </div>
          
          <div className="max-w-4xl mx-auto grid grid-cols-1 md:grid-cols-3 gap-6">
            {/* Download Application */}
            <div className="bg-gradient-to-br from-blue-50 to-blue-100 rounded-2xl p-6 text-center hover:shadow-lg transition-all duration-300 cursor-pointer">
              <div className="w-16 h-16 bg-blue-500 rounded-full flex items-center justify-center mx-auto mb-4">
                <Download className="h-8 w-8 text-white" />
              </div>
              <h3 className="text-xl font-semibold text-gray-900 mb-2">Download Application</h3>
              <p className="text-gray-600 mb-4">Get the complete SportsPro setup package with installation guides</p>
              <Button 
                onClick={handleDownloadApplication}
                className="bg-blue-500 hover:bg-blue-600 text-white w-full"
              >
                Download Now
              </Button>
            </div>

            {/* Environment Template */}
            <div className="bg-gradient-to-br from-green-50 to-green-100 rounded-2xl p-6 text-center hover:shadow-lg transition-all duration-300 cursor-pointer">
              <div className="w-16 h-16 bg-green-500 rounded-full flex items-center justify-center mx-auto mb-4">
                <FileText className="h-8 w-8 text-white" />
              </div>
              <h3 className="text-xl font-semibold text-gray-900 mb-2">Environment Template</h3>
              <p className="text-gray-600 mb-4">Pre-configured environment settings for quick deployment</p>
              <Button 
                onClick={handleDownloadTemplate}
                className="bg-green-500 hover:bg-green-600 text-white w-full"
              >
                Get Template
              </Button>
            </div>

            {/* Configuration Examples */}
            <div className="bg-gradient-to-br from-purple-50 to-purple-100 rounded-2xl p-6 text-center hover:shadow-lg transition-all duration-300 cursor-pointer">
              <div className="w-16 h-16 bg-purple-500 rounded-full flex items-center justify-center mx-auto mb-4">
                <Settings className="h-8 w-8 text-white" />
              </div>
              <h3 className="text-xl font-semibold text-gray-900 mb-2">Configuration Examples</h3>
              <p className="text-gray-600 mb-4">Sample configurations for different sports and team sizes</p>
              <Button 
                onClick={handleDownloadConfigs}
                className="bg-purple-500 hover:bg-purple-600 text-white w-full"
              >
                View Examples
              </Button>
            </div>
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="py-20 bg-gradient-to-r from-primary to-purple-600">
        <div className="container mx-auto px-4 text-center">
          <div className="max-w-3xl mx-auto">
            <h2 className="text-2xl sm:text-3xl lg:text-4xl font-bold text-white mb-6">
              Ready to Transform Your Sports Management?
            </h2>
            <p className="text-lg sm:text-xl text-white/90 mb-8">
              Join thousands of teams worldwide. Start your free 14-day trial today.
            </p>
            
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Button 
                size="lg" 
                onClick={() => window.location.href = "/auth"}
                className="bg-white text-primary hover:bg-gray-50 text-base sm:text-lg px-6 sm:px-8 py-3 sm:py-4 shadow-xl"
              >
                Start Free Trial
                <ArrowRight className="ml-2 h-4 w-4 sm:h-5 sm:w-5" />
              </Button>
              <Button 
                size="lg" 
                variant="outline"
                className="border-2 border-white text-white hover:bg-white/10 text-base sm:text-lg px-6 sm:px-8 py-3 sm:py-4"
              >
                <Headphones className="mr-2 h-4 w-4 sm:h-5 sm:w-5" />
                Talk to Sales
              </Button>
            </div>
            
            <p className="text-white/70 text-sm mt-4">
              No credit card required • 14-day free trial • Cancel anytime
            </p>
          </div>
        </div>
      </section>

      {/* Footer */}
      <footer className="bg-gray-900 text-white py-12 sm:py-16">
        <div className="container mx-auto px-4">
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6 sm:gap-8">
            <div className="space-y-4 sm:col-span-2 lg:col-span-1">
              <div className="flex items-center space-x-3">
                <div className="w-10 h-10 bg-gradient-to-br from-primary to-purple-600 rounded-lg flex items-center justify-center">
                  <Trophy className="h-6 w-6 text-white" />
                </div>
                <span className="text-lg sm:text-xl font-bold">SportsPro</span>
              </div>
              <p className="text-sm sm:text-base text-gray-400">
                Professional sports management platform trusted by teams worldwide.
              </p>
            </div>
            
            <div>
              <h3 className="text-sm sm:text-base font-semibold mb-3 sm:mb-4">Product</h3>
              <ul className="space-y-2 text-gray-400">
                <li><a href="/features" className="text-sm sm:text-base hover:text-white transition-colors">Features</a></li>
                <li><a href="/pricing" className="text-sm sm:text-base hover:text-white transition-colors">Pricing</a></li>
                <li><a href="/security" className="text-sm sm:text-base hover:text-white transition-colors">Security</a></li>
                <li><a href="/integrations" className="text-sm sm:text-base hover:text-white transition-colors">Integrations</a></li>
              </ul>
            </div>
            
            <div>
              <h3 className="text-sm sm:text-base font-semibold mb-3 sm:mb-4">Support</h3>
              <ul className="space-y-2 text-gray-400">
                <li><a href="/docs" className="text-sm sm:text-base hover:text-white transition-colors">Documentation</a></li>
                <li><a href="/help" className="text-sm sm:text-base hover:text-white transition-colors">Help Center</a></li>
                <li><a href="/contact" className="text-sm sm:text-base hover:text-white transition-colors">Contact</a></li>
                <li><a href="/status" className="text-sm sm:text-base hover:text-white transition-colors">Status</a></li>
              </ul>
            </div>
            
            <div>
              <h3 className="text-sm sm:text-base font-semibold mb-3 sm:mb-4">Company</h3>
              <ul className="space-y-2 text-gray-400">
                <li><a href="/about" className="text-sm sm:text-base hover:text-white transition-colors">About</a></li>
                <li><a href="/careers" className="text-sm sm:text-base hover:text-white transition-colors">Careers</a></li>
                <li><a href="/press" className="text-sm sm:text-base hover:text-white transition-colors">Press</a></li>
                <li><a href="/partners" className="text-sm sm:text-base hover:text-white transition-colors">Partners</a></li>
              </ul>
            </div>
          </div>
          
          <div className="border-t border-gray-800 mt-8 sm:mt-12 pt-6 sm:pt-8 flex flex-col sm:flex-row justify-between items-center">
            <p className="text-gray-400 text-xs sm:text-sm">
              © 2025 SportsPro. All rights reserved.
            </p>
            <div className="flex flex-wrap justify-center space-x-4 sm:space-x-6 mt-4 sm:mt-0">
              <a href="/privacy" className="text-gray-400 hover:text-white text-xs sm:text-sm transition-colors">Privacy</a>
              <a href="/terms" className="text-gray-400 hover:text-white text-xs sm:text-sm transition-colors">Terms</a>
              <a href="/cookies" className="text-gray-400 hover:text-white text-xs sm:text-sm transition-colors">Cookies</a>
            </div>
          </div>
        </div>
      </footer>
    </div>
  );
}