import { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Accordion, AccordionContent, AccordionItem, AccordionTrigger } from "@/components/ui/accordion";
import { 
  Search, BookOpen, Users, UserPlus, Trophy, BarChart3, 
  Calendar, MessageSquare, Settings, Shield, CreditCard,
  Globe, PlayCircle, Download, ExternalLink, ChevronRight,
  Video, FileText, HelpCircle, Mail, Phone, Clock, Target,
  MousePointer2, Zap, TrendingUp, Award, Eye
} from "lucide-react";
import { useLocation } from "wouter";
import dashboardScreenshot from "@/assets/screenshots/dashboard.svg";
import teamsScreenshot from "@/assets/screenshots/teams.svg";
import playersScreenshot from "@/assets/screenshots/players.svg";
import rosterScreenshot from "@/assets/screenshots/roster.svg";

export default function Help() {
  const [searchQuery, setSearchQuery] = useState("");
  const [, setLocation] = useLocation();

  const quickStartGuides = [
    {
      title: "Dashboard Overview",
      description: "Understanding your sports management dashboard",
      icon: <BarChart3 className="h-6 w-6 text-blue-600" />,
      duration: "3 min read",
      screenshot: dashboardScreenshot,
      steps: [
        "View real-time team statistics and key metrics",
        "Monitor attendance rates and player activity",
        "Track recent team activities and notifications",
        "Access quick actions for common tasks"
      ]
    },
    {
      title: "Team Management",
      description: "Create and organize your sports teams",
      icon: <Trophy className="h-6 w-6 text-orange-600" />,
      duration: "5 min read",
      screenshot: teamsScreenshot,
      steps: [
        "Click 'Add Team' to create a new team",
        "Upload team logo and set basic information",
        "Configure sport type and team size",
        "View teams with beautiful gradient headers"
      ]
    },
    {
      title: "Player Registration",
      description: "Register and manage player profiles",
      icon: <Users className="h-6 w-6 text-green-600" />,
      duration: "6 min read",
      screenshot: playersScreenshot,
      steps: [
        "Navigate to Players section and click 'Register Player'",
        "Fill in comprehensive player details and contact info",
        "Upload profile photos and assign jersey numbers",
        "Set positions, status, and team assignments"
      ]
    },
    {
      title: "Roster Management",
      description: "Organize players with drag-and-drop interface",
      icon: <Target className="h-6 w-6 text-purple-600" />,
      duration: "4 min read",
      screenshot: rosterScreenshot,
      steps: [
        "Select your team from the roster dropdown",
        "Drag players from available list to field positions",
        "Choose formation type (15s, 7s, or 10s Rugby)",
        "Save roster configuration for matches"
      ]
    },
    {
      title: "Multi-Sport Support",
      description: "Managing different sports in one platform",
      icon: <Globe className="h-6 w-6 text-cyan-600" />,
      duration: "4 min read",
      steps: [
        "Support for Rugby, Soccer, Basketball, Cricket, Baseball",
        "Hockey, Netball, Volleyball, Badminton, and Football",
        "Sport-specific positions and team configurations",
        "Unified management across all sports"
      ]
    }
  ];

  const featureGuides = [
    {
      category: "Team Management",
      icon: <Users className="h-5 w-5" />,
      items: [
        {
          title: "Creating and Managing Teams",
          description: "Learn how to set up teams, add players, and manage team information",
          sections: [
            "Setting up a new team",
            "Adding team details and logo",
            "Managing team roster",
            "Team statistics overview"
          ]
        },
        {
          title: "Player Registration Process",
          description: "Step-by-step guide to registering new players",
          sections: [
            "Player information form",
            "Document upload requirements",
            "Medical information",
            "Emergency contact details"
          ]
        },
        {
          title: "Coaching Staff Management",
          description: "How to add and manage coaching staff",
          sections: [
            "Adding new coaches",
            "Setting coach roles and responsibilities",
            "Coach profile management",
            "Performance tracking for coaches"
          ]
        }
      ]
    },
    {
      category: "Performance Analytics",
      icon: <BarChart3 className="h-5 w-5" />,
      items: [
        {
          title: "Player Statistics",
          description: "Track and analyze individual player performance",
          sections: [
            "Recording match statistics",
            "Performance trend analysis",
            "Comparative player metrics",
            "Generating performance reports"
          ]
        },
        {
          title: "Team Analytics",
          description: "Monitor team performance and progress",
          sections: [
            "Team performance dashboard",
            "Match result tracking",
            "Season progress monitoring",
            "Tactical analysis tools"
          ]
        }
      ]
    },
    {
      category: "Communication",
      icon: <MessageSquare className="h-5 w-5" />,
      items: [
        {
          title: "Team Communications",
          description: "Keep your team informed and connected",
          sections: [
            "Sending announcements",
            "Managing team news",
            "Event notifications",
            "Direct messaging features"
          ]
        },
        {
          title: "Parent Portal",
          description: "Enable parent access and communication",
          sections: [
            "Setting up parent accounts",
            "Sharing player updates",
            "Event scheduling for parents",
            "Permission and consent management"
          ]
        }
      ]
    },
    {
      category: "Administration",
      icon: <Settings className="h-5 w-5" />,
      items: [
        {
          title: "User Management",
          description: "Manage user accounts and permissions",
          sections: [
            "Creating user accounts",
            "Setting user roles and permissions",
            "Managing access levels",
            "Account security settings"
          ]
        },
        {
          title: "Organization Settings",
          description: "Configure your organization preferences",
          sections: [
            "Organization profile setup",
            "Subscription management",
            "Security settings",
            "Data export and backup"
          ]
        }
      ]
    }
  ];

  const faqs = [
    {
      question: "How do I add a new player to my team?",
      answer: "Navigate to the Players section and click 'Register New Player'. Fill in the required information including personal details, position, and emergency contacts. You can also upload a profile photo for the player."
    },
    {
      question: "Can I manage multiple teams within one organization?",
      answer: "Yes, SportSync supports multiple teams within a single organization. You can create different teams for various age groups, skill levels, or sports disciplines."
    },
    {
      question: "How do I track player performance statistics?",
      answer: "Use the Performance Analytics section to record match statistics, training metrics, and other performance indicators. The system automatically generates charts and reports based on this data."
    },
    {
      question: "What roles and permissions are available?",
      answer: "SportSync offers several role levels: Super Admin (full access), Admin (organization management), Coach (team management), and Player/Parent (limited access). Each role has specific permissions tailored to their responsibilities."
    },
    {
      question: "How do I send communications to my team?",
      answer: "Use the Communication Hub to send announcements, share news, and notify team members about events. You can target specific groups or send messages to the entire team."
    },
    {
      question: "Can parents access their child's information?",
      answer: "Yes, parents can be given access to view their child's profile, performance statistics, and team communications through the parent portal feature."
    },
    {
      question: "How do I backup my team data?",
      answer: "Navigate to Organization Settings and use the Data Export feature to create backups of your team information, player profiles, and performance data."
    },
    {
      question: "What sports are supported by the platform?",
      answer: "SportSync supports a wide range of sports including football, basketball, rugby, soccer, baseball, and many others. The platform is flexible enough to accommodate most team sports."
    }
  ];

  const filteredFAQs = faqs.filter(faq =>
    faq.question.toLowerCase().includes(searchQuery.toLowerCase()) ||
    faq.answer.toLowerCase().includes(searchQuery.toLowerCase())
  );

  return (
    <div className="container mx-auto px-4 py-8 max-w-6xl">
      {/* Header */}
      <div className="text-center mb-8">
        <div className="flex items-center justify-center mb-4">
          <div className="bg-blue-100 p-3 rounded-full">
            <BookOpen className="h-8 w-8 text-blue-600" />
          </div>
        </div>
        <h1 className="text-3xl font-bold text-gray-900 mb-2">
          Help & Documentation
        </h1>
        <p className="text-gray-600 max-w-2xl mx-auto">
          Everything you need to know about using SportSync to manage your teams effectively
        </p>
      </div>

      {/* Search Bar */}
      <div className="max-w-md mx-auto mb-8">
        <div className="relative">
          <Search className="absolute left-3 top-3 h-4 w-4 text-gray-400" />
          <Input
            placeholder="Search documentation..."
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="pl-10"
          />
        </div>
      </div>

      <Tabs defaultValue="getting-started" className="space-y-6">
        <TabsList className="grid w-full grid-cols-4">
          <TabsTrigger value="getting-started">Getting Started</TabsTrigger>
          <TabsTrigger value="features">Features</TabsTrigger>
          <TabsTrigger value="faqs">FAQs</TabsTrigger>
          <TabsTrigger value="support">Support</TabsTrigger>
        </TabsList>

        {/* Getting Started Tab */}
        <TabsContent value="getting-started" className="space-y-6">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {quickStartGuides.map((guide, index) => (
              <Card key={index} className="hover:shadow-lg transition-shadow">
                <CardHeader>
                  <div className="flex items-center justify-between mb-2">
                    {guide.icon}
                    <Badge variant="secondary">{guide.duration}</Badge>
                  </div>
                  <CardTitle className="text-lg">{guide.title}</CardTitle>
                  <CardDescription>{guide.description}</CardDescription>
                </CardHeader>
                <CardContent>
                  <ul className="space-y-2">
                    {guide.steps.map((step, stepIndex) => (
                      <li key={stepIndex} className="flex items-center text-sm text-gray-600">
                        <div className="w-4 h-4 bg-blue-100 rounded-full flex items-center justify-center mr-2 flex-shrink-0">
                          <span className="text-xs text-blue-600 font-medium">{stepIndex + 1}</span>
                        </div>
                        {step}
                      </li>
                    ))}
                  </ul>
                  <Button 
                    variant="outline" 
                    className="w-full mt-4"
                    onClick={() => setLocation("/onboarding")}
                  >
                    <PlayCircle className="h-4 w-4 mr-2" />
                    Start Guide
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>

          {/* Quick Actions */}
          <Card>
            <CardHeader>
              <CardTitle>Quick Actions</CardTitle>
              <CardDescription>
                Jump straight to common tasks
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                <Button 
                  variant="outline" 
                  className="h-auto p-4 flex flex-col items-center space-y-2"
                  onClick={() => setLocation("/onboarding")}
                >
                  <PlayCircle className="h-6 w-6 text-blue-600" />
                  <span className="text-sm">Take Tour</span>
                </Button>
                <Button 
                  variant="outline" 
                  className="h-auto p-4 flex flex-col items-center space-y-2"
                  onClick={() => setLocation("/teams")}
                >
                  <Users className="h-6 w-6 text-green-600" />
                  <span className="text-sm">Manage Teams</span>
                </Button>
                <Button 
                  variant="outline" 
                  className="h-auto p-4 flex flex-col items-center space-y-2"
                  onClick={() => setLocation("/players/register")}
                >
                  <UserPlus className="h-6 w-6 text-purple-600" />
                  <span className="text-sm">Add Player</span>
                </Button>
                <Button 
                  variant="outline" 
                  className="h-auto p-4 flex flex-col items-center space-y-2"
                  onClick={() => setLocation("/settings")}
                >
                  <Settings className="h-6 w-6 text-orange-600" />
                  <span className="text-sm">Settings</span>
                </Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        {/* Features Tab */}
        <TabsContent value="features" className="space-y-6">
          {featureGuides.map((category, categoryIndex) => (
            <Card key={categoryIndex}>
              <CardHeader>
                <div className="flex items-center space-x-2">
                  {category.icon}
                  <CardTitle>{category.category}</CardTitle>
                </div>
              </CardHeader>
              <CardContent>
                <Accordion type="single" collapsible className="w-full">
                  {category.items.map((item, itemIndex) => (
                    <AccordionItem key={itemIndex} value={`${categoryIndex}-${itemIndex}`}>
                      <AccordionTrigger className="text-left">
                        <div>
                          <div className="font-medium">{item.title}</div>
                          <div className="text-sm text-gray-500 mt-1">{item.description}</div>
                        </div>
                      </AccordionTrigger>
                      <AccordionContent>
                        <div className="space-y-2">
                          {item.sections.map((section, sectionIndex) => (
                            <div key={sectionIndex} className="flex items-center p-2 hover:bg-gray-50 rounded cursor-pointer">
                              <FileText className="h-4 w-4 text-gray-400 mr-2" />
                              <span className="text-sm">{section}</span>
                              <ChevronRight className="h-4 w-4 text-gray-400 ml-auto" />
                            </div>
                          ))}
                        </div>
                      </AccordionContent>
                    </AccordionItem>
                  ))}
                </Accordion>
              </CardContent>
            </Card>
          ))}
        </TabsContent>

        {/* FAQs Tab */}
        <TabsContent value="faqs" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>Frequently Asked Questions</CardTitle>
              <CardDescription>
                Find answers to common questions about using SportSync
              </CardDescription>
            </CardHeader>
            <CardContent>
              <Accordion type="single" collapsible className="w-full">
                {filteredFAQs.map((faq, index) => (
                  <AccordionItem key={index} value={`faq-${index}`}>
                    <AccordionTrigger className="text-left">
                      <div className="flex items-center">
                        <HelpCircle className="h-4 w-4 text-blue-600 mr-2 flex-shrink-0" />
                        {faq.question}
                      </div>
                    </AccordionTrigger>
                    <AccordionContent>
                      <div className="pl-6 text-gray-600">
                        {faq.answer}
                      </div>
                    </AccordionContent>
                  </AccordionItem>
                ))}
              </Accordion>
            </CardContent>
          </Card>
        </TabsContent>

        {/* Support Tab */}
        <TabsContent value="support" className="space-y-6">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <Card>
              <CardHeader>
                <CardTitle>Contact Support</CardTitle>
                <CardDescription>
                  Get help from our support team
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center p-3 border rounded-lg">
                  <Mail className="h-5 w-5 text-blue-600 mr-3" />
                  <div>
                    <div className="font-medium">Email Support</div>
                    <div className="text-sm text-gray-600">support@sportsync.com</div>
                  </div>
                </div>
                <div className="flex items-center p-3 border rounded-lg">
                  <Phone className="h-5 w-5 text-green-600 mr-3" />
                  <div>
                    <div className="font-medium">Phone Support</div>
                    <div className="text-sm text-gray-600">+1 (555) 123-4567</div>
                  </div>
                </div>
                <div className="flex items-center p-3 border rounded-lg">
                  <Clock className="h-5 w-5 text-orange-600 mr-3" />
                  <div>
                    <div className="font-medium">Business Hours</div>
                    <div className="text-sm text-gray-600">Mon-Fri: 9AM - 6PM EST</div>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Additional Resources</CardTitle>
                <CardDescription>
                  More ways to get help and learn
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <Button variant="outline" className="w-full justify-start">
                  <Video className="h-4 w-4 mr-2" />
                  Video Tutorials
                  <ExternalLink className="h-4 w-4 ml-auto" />
                </Button>
                <Button variant="outline" className="w-full justify-start">
                  <Download className="h-4 w-4 mr-2" />
                  Download User Manual
                  <ExternalLink className="h-4 w-4 ml-auto" />
                </Button>
                <Button variant="outline" className="w-full justify-start">
                  <MessageSquare className="h-4 w-4 mr-2" />
                  Community Forum
                  <ExternalLink className="h-4 w-4 ml-auto" />
                </Button>
                <Button variant="outline" className="w-full justify-start">
                  <Globe className="h-4 w-4 mr-2" />
                  Knowledge Base
                  <ExternalLink className="h-4 w-4 ml-auto" />
                </Button>
              </CardContent>
            </Card>
          </div>

          {/* System Status */}
          <Card>
            <CardHeader>
              <CardTitle>System Status</CardTitle>
              <CardDescription>
                Current status of SportSync services
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                <div className="flex items-center justify-between p-3 bg-green-50 rounded-lg">
                  <div className="flex items-center">
                    <div className="w-3 h-3 bg-green-500 rounded-full mr-3"></div>
                    <span className="font-medium">Core Platform</span>
                  </div>
                  <Badge variant="outline" className="text-green-700 border-green-200">
                    Operational
                  </Badge>
                </div>
                <div className="flex items-center justify-between p-3 bg-green-50 rounded-lg">
                  <div className="flex items-center">
                    <div className="w-3 h-3 bg-green-500 rounded-full mr-3"></div>
                    <span className="font-medium">Player Management</span>
                  </div>
                  <Badge variant="outline" className="text-green-700 border-green-200">
                    Operational
                  </Badge>
                </div>
                <div className="flex items-center justify-between p-3 bg-green-50 rounded-lg">
                  <div className="flex items-center">
                    <div className="w-3 h-3 bg-green-500 rounded-full mr-3"></div>
                    <span className="font-medium">Communication Services</span>
                  </div>
                  <Badge variant="outline" className="text-green-700 border-green-200">
                    Operational
                  </Badge>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}