import { useParams, Link } from "wouter";
import { useQuery } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { ArrowLeft, Mail, Phone, Calendar, Award, Users, Clock } from "lucide-react";
import type { Coach } from "@shared/schema";

export default function CoachDetail() {
  const { id } = useParams();
  const coachId = parseInt(id || "0");

  const { data: coach, isLoading } = useQuery({
    queryKey: [`/api/coaches/${coachId}`],
    enabled: !!coachId && !isNaN(coachId),
  });

  if (isLoading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-primary mx-auto"></div>
          <p className="mt-4 text-gray-600">Loading coach details...</p>
        </div>
      </div>
    );
  }

  if (!coach) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <h2 className="text-2xl font-bold text-gray-900 mb-2">Coach not found</h2>
          <p className="text-gray-600 mb-4">The coach you're looking for doesn't exist.</p>
          <Link to="/coaches">
            <Button>
              <ArrowLeft className="h-4 w-4 mr-2" />
              Back to Coaches
            </Button>
          </Link>
        </div>
      </div>
    );
  }

  const coachData = coach as Coach;

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Beautiful Gradient Header */}
      <div className="relative bg-gradient-to-br from-blue-600 via-purple-600 to-blue-800 text-white overflow-hidden">
        {/* Background decorative elements */}
        <div className="absolute inset-0">
          <div className="absolute top-0 left-1/4 w-72 h-72 bg-white/10 rounded-full blur-3xl"></div>
          <div className="absolute bottom-0 right-1/4 w-96 h-96 bg-purple-300/20 rounded-full blur-3xl"></div>
          <div className="absolute top-1/2 left-3/4 w-64 h-64 bg-blue-300/10 rounded-full blur-2xl"></div>
        </div>
        
        {/* Header content */}
        <div className="relative z-10 px-6 py-12">
          <div className="max-w-7xl mx-auto">
            {/* Back button */}
            <div className="mb-6">
              <Link to="/coaches">
                <Button variant="ghost" className="text-white hover:bg-white/10">
                  <ArrowLeft className="h-4 w-4 mr-2" />
                  Back to Coaches
                </Button>
              </Link>
            </div>

            <div className="flex items-center space-x-6">
              {/* Coach Avatar */}
              <div className="w-24 h-24 bg-white/20 backdrop-blur-sm rounded-full p-3 flex items-center justify-center overflow-hidden">
                {coachData.profileImageUrl ? (
                  <img 
                    src={coachData.profileImageUrl} 
                    alt={`${coachData.firstName} ${coachData.lastName}`}
                    className="w-full h-full object-cover rounded-full"
                  />
                ) : (
                  <div className="w-full h-full bg-white/30 rounded-full flex items-center justify-center text-white text-2xl font-bold">
                    {coachData.firstName[0]}{coachData.lastName[0]}
                  </div>
                )}
              </div>

              <div className="flex-1">
                <h1 className="text-4xl font-bold mb-2">
                  {coachData.firstName} {coachData.lastName}
                </h1>
                <div className="flex items-center gap-4 text-lg text-white/90">
                  <Badge variant="secondary" className="bg-white/20 text-white border-white/30">
                    {coachData.specialization || "Coach"}
                  </Badge>
                  <Badge variant="outline" className="border-white/30 text-white">
                    {coachData.status}
                  </Badge>
                </div>
                {coachData.experience && (
                  <p className="mt-3 text-white/80">
                    {coachData.experience} years of coaching experience
                  </p>
                )}
              </div>

              {/* Quick Stats */}
              <div className="hidden lg:flex flex-col gap-3">
                <div className="bg-white/10 backdrop-blur-sm rounded-lg p-4 text-center min-w-[100px]">
                  <div className="text-2xl font-bold">{coachData.experience || 0}</div>
                  <div className="text-sm text-white/80">Years Exp.</div>
                </div>
                <div className="bg-white/10 backdrop-blur-sm rounded-lg p-4 text-center min-w-[100px]">
                  <div className="text-2xl font-bold">{coachData.status === 'active' ? 'Active' : 'Inactive'}</div>
                  <div className="text-sm text-white/80">Status</div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Main Content */}
      <div className="p-6">
        <div className="max-w-7xl mx-auto">
          {/* Coach Information Cards */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Status</CardTitle>
                <Users className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold capitalize">{coachData.status}</div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Experience</CardTitle>
                <Award className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">
                  {coachData.experience || 0} Years
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Joined</CardTitle>
                <Calendar className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-lg font-bold">
                  {new Date(coachData.createdAt!).toLocaleDateString()}
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Specialization</CardTitle>
                <Clock className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-lg font-bold">
                  {coachData.specialization || "General"}
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Detailed Information */}
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {/* Contact Information */}
            <Card>
              <CardHeader>
                <CardTitle>Contact Information</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                {coachData.email && (
                  <div className="flex items-center space-x-3">
                    <Mail className="h-5 w-5 text-gray-400" />
                    <div>
                      <p className="text-sm font-medium text-gray-600">Email</p>
                      <p className="text-gray-900">{coachData.email}</p>
                    </div>
                  </div>
                )}
                {coachData.phone && (
                  <div className="flex items-center space-x-3">
                    <Phone className="h-5 w-5 text-gray-400" />
                    <div>
                      <p className="text-sm font-medium text-gray-600">Phone</p>
                      <p className="text-gray-900">{coachData.phone}</p>
                    </div>
                  </div>
                )}
                <div className="flex items-center space-x-3">
                  <Calendar className="h-5 w-5 text-gray-400" />
                  <div>
                    <p className="text-sm font-medium text-gray-600">Joined</p>
                    <p className="text-gray-900">{new Date(coachData.createdAt!).toLocaleDateString()}</p>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Professional Information */}
            <Card>
              <CardHeader>
                <CardTitle>Professional Details</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center space-x-3">
                  <Award className="h-5 w-5 text-gray-400" />
                  <div>
                    <p className="text-sm font-medium text-gray-600">Specialization</p>
                    <p className="text-gray-900">{coachData.specialization || "General Coaching"}</p>
                  </div>
                </div>
                <div className="flex items-center space-x-3">
                  <Clock className="h-5 w-5 text-gray-400" />
                  <div>
                    <p className="text-sm font-medium text-gray-600">Experience</p>
                    <p className="text-gray-900">{coachData.experience || 0} years</p>
                  </div>
                </div>
                <div className="flex items-center space-x-3">
                  <Users className="h-5 w-5 text-gray-400" />
                  <div>
                    <p className="text-sm font-medium text-gray-600">Status</p>
                    <Badge variant={coachData.status === "active" ? "default" : "secondary"} className="mt-1">
                      {coachData.status}
                    </Badge>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </div>
  );
}