import React, { useState, useEffect } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { 
  ChevronRight, 
  ChevronLeft, 
  X, 
  Users, 
  UserPlus, 
  Award, 
  BarChart3, 
  Fingerprint,
  CheckCircle,
  Lightbulb,
  Target,
  Star
} from "lucide-react";
import { useLocation } from "wouter";

interface OnboardingStep {
  id: string;
  title: string;
  description: string;
  icon: React.ComponentType<any>;
  targetElement?: string;
  route?: string;
  tip: string;
  action?: string;
}

const onboardingSteps: OnboardingStep[] = [
  {
    id: "welcome",
    title: "Welcome to SportsPro!",
    description: "Your complete sports management solution. Let's take a quick tour to get you started with managing teams, players, and coaches effectively.",
    icon: Star,
    tip: "This walkthrough will take about 2 minutes and show you the most important features.",
    action: "Start Tour"
  },
  {
    id: "dashboard",
    title: "Dashboard Overview",
    description: "Your central hub for monitoring team performance, player statistics, and quick actions. View real-time analytics and get insights at a glance.",
    icon: BarChart3,
    route: "/",
    targetElement: "[data-onboarding='dashboard-stats']",
    tip: "The dashboard shows live statistics and updates automatically when you add new data.",
    action: "Explore Dashboard"
  },
  {
    id: "teams",
    title: "Team Management",
    description: "Create and manage your teams here. Add team logos, descriptions, and track their performance across different sports.",
    icon: Users,
    route: "/teams",
    targetElement: "[data-onboarding='teams-section']",
    tip: "You can assign multiple coaches to each team and track their specializations.",
    action: "View Teams"
  },
  {
    id: "players",
    title: "Player Profiles",
    description: "Manage detailed player profiles including personal information, statistics, photos, and performance metrics.",
    icon: UserPlus,
    route: "/players",
    targetElement: "[data-onboarding='players-section']",
    tip: "Upload player photos and track their progress with detailed statistics and biometric data.",
    action: "See Players"
  },
  {
    id: "coaches",
    title: "Coach Management",
    description: "Handle coach assignments, track their experience, and manage their specializations across multiple teams.",
    icon: Award,
    route: "/coaches",
    targetElement: "[data-onboarding='coaches-section']",
    tip: "Coaches can work with multiple teams and have different roles like head coach or assistant.",
    action: "View Coaches"
  },
  {
    id: "biometric",
    title: "Biometric Integration",
    description: "Advanced biometric scanning and data collection for enhanced player identification and performance tracking.",
    icon: Fingerprint,
    route: "/biometric",
    targetElement: "[data-onboarding='biometric-section']",
    tip: "Use fingerprint scanning for secure player identification and attendance tracking.",
    action: "Explore Biometrics"
  },
  {
    id: "complete",
    title: "You're All Set!",
    description: "Congratulations! You've completed the tour. Start by creating your first team or adding players to begin managing your sports organization.",
    icon: CheckCircle,
    tip: "Remember, you can always return to any section using the sidebar navigation.",
    action: "Get Started"
  }
];

interface OnboardingWalkthroughProps {
  isVisible: boolean;
  onComplete: () => void;
  onSkip: () => void;
}

export default function OnboardingWalkthrough({ isVisible, onComplete, onSkip }: OnboardingWalkthroughProps) {
  const [currentStep, setCurrentStep] = useState(0);
  const [, setLocation] = useLocation();

  const step = onboardingSteps[currentStep];
  const progress = ((currentStep + 1) / onboardingSteps.length) * 100;

  useEffect(() => {
    if (isVisible && step.route && currentStep > 0) {
      setLocation(step.route);
    }
  }, [currentStep, isVisible, step.route, setLocation]);

  const handleNext = () => {
    if (currentStep < onboardingSteps.length - 1) {
      setCurrentStep(currentStep + 1);
    } else {
      onComplete();
    }
  };

  const handlePrevious = () => {
    if (currentStep > 0) {
      setCurrentStep(currentStep - 1);
    }
  };

  const handleSkip = () => {
    onSkip();
  };

  if (!isVisible) return null;

  return (
    <div className="fixed inset-0 z-50 bg-black/50 backdrop-blur-sm flex items-center justify-center p-4">
      <Card className="w-full max-w-2xl shadow-2xl border-2 border-primary/20">
        <CardHeader className="text-center space-y-4">
          <div className="flex items-center justify-between">
            <Badge variant="secondary" className="text-xs">
              Step {currentStep + 1} of {onboardingSteps.length}
            </Badge>
            <Button
              variant="ghost"
              size="sm"
              onClick={handleSkip}
              className="text-gray-500 hover:text-gray-700"
            >
              <X className="h-4 w-4" />
            </Button>
          </div>
          
          <Progress value={progress} className="w-full h-2" />
          
          <div className="flex flex-col items-center space-y-4">
            <div className="w-16 h-16 bg-primary/10 rounded-full flex items-center justify-center">
              <step.icon className="h-8 w-8 text-primary" />
            </div>
            
            <CardTitle className="text-2xl font-bold text-gray-900">
              {step.title}
            </CardTitle>
          </div>
        </CardHeader>

        <CardContent className="space-y-6">
          <CardDescription className="text-center text-lg leading-relaxed text-gray-600">
            {step.description}
          </CardDescription>

          <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
            <div className="flex items-start space-x-3">
              <Lightbulb className="h-5 w-5 text-blue-600 mt-0.5 flex-shrink-0" />
              <div>
                <p className="text-sm font-medium text-blue-900">Pro Tip</p>
                <p className="text-sm text-blue-700">{step.tip}</p>
              </div>
            </div>
          </div>

          {step.id === "complete" && (
            <div className="bg-green-50 border border-green-200 rounded-lg p-4">
              <div className="flex items-start space-x-3">
                <Target className="h-5 w-5 text-green-600 mt-0.5 flex-shrink-0" />
                <div>
                  <p className="text-sm font-medium text-green-900">Next Steps</p>
                  <ul className="text-sm text-green-700 mt-1 space-y-1">
                    <li>• Create your first team</li>
                    <li>• Add player profiles with photos</li>
                    <li>• Assign coaches to teams</li>
                    <li>• Set up biometric scanning</li>
                  </ul>
                </div>
              </div>
            </div>
          )}

          <div className="flex items-center justify-between pt-4">
            <Button
              variant="outline"
              onClick={handlePrevious}
              disabled={currentStep === 0}
              className="flex items-center space-x-2"
            >
              <ChevronLeft className="h-4 w-4" />
              <span>Previous</span>
            </Button>

            <div className="flex space-x-2">
              <Button
                variant="ghost"
                onClick={handleSkip}
                className="text-gray-500"
              >
                Skip Tour
              </Button>
              
              <Button
                onClick={handleNext}
                className="flex items-center space-x-2 bg-primary hover:bg-primary/90"
              >
                <span>{step.action || "Next"}</span>
                {currentStep === onboardingSteps.length - 1 ? (
                  <CheckCircle className="h-4 w-4" />
                ) : (
                  <ChevronRight className="h-4 w-4" />
                )}
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}