import { useState } from "react";
import { BookOpen, Play, CheckCircle, RotateCcw, HelpCircle } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";
import { useOnboarding } from "./onboarding-provider";

interface OnboardingMenuProps {
  className?: string;
}

const TOUR_DESCRIPTIONS = {
  dashboard: {
    title: "Dashboard Overview",
    description: "Learn the basics of navigating your SportsPro dashboard",
    duration: "2-3 minutes",
    steps: 4,
  },
  teamManagement: {
    title: "Team Management",
    description: "Create teams, manage rosters, and organize your players",
    duration: "3-4 minutes",
    steps: 3,
  },
  playerRegistration: {
    title: "Player Registration",
    description: "Register new players and manage their profiles",
    duration: "2-3 minutes",
    steps: 4,
  },
};

export function OnboardingMenu({ className }: OnboardingMenuProps) {
  const [isOpen, setIsOpen] = useState(false);
  const { startOnboarding, completedSteps, isStepCompleted } = useOnboarding();

  const getTourProgress = (tourType: string) => {
    const tourSteps = TOUR_DESCRIPTIONS[tourType as keyof typeof TOUR_DESCRIPTIONS];
    if (!tourSteps) return { completed: 0, total: 0, isCompleted: false };
    
    // This is a simplified check - in a real app you'd track tour-specific completion
    const completed = completedSteps.length;
    const total = tourSteps.steps;
    const isCompleted = completed >= total;
    
    return { completed, total, isCompleted };
  };

  const handleStartTour = (tourType: string) => {
    startOnboarding(tourType);
    setIsOpen(false);
  };

  return (
    <Dialog open={isOpen} onOpenChange={setIsOpen}>
      <DialogTrigger asChild>
        <Button
          variant="outline"
          size="sm"
          className={`gap-2 ${className}`}
        >
          <HelpCircle className="h-4 w-4" />
          Help & Tutorials
        </Button>
      </DialogTrigger>
      
      <DialogContent className="max-w-2xl">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <BookOpen className="h-5 w-5" />
            Interactive Tutorials
          </DialogTitle>
        </DialogHeader>
        
        <div className="grid gap-4 mt-4">
          {Object.entries(TOUR_DESCRIPTIONS).map(([tourType, tour]) => {
            const progress = getTourProgress(tourType);
            
            return (
              <Card key={tourType} className="relative">
                <CardHeader className="pb-3">
                  <div className="flex items-center justify-between">
                    <CardTitle className="text-lg">{tour.title}</CardTitle>
                    {progress.isCompleted && (
                      <Badge variant="secondary" className="gap-1">
                        <CheckCircle className="h-3 w-3" />
                        Completed
                      </Badge>
                    )}
                  </div>
                  <p className="text-sm text-muted-foreground">
                    {tour.description}
                  </p>
                </CardHeader>
                
                <CardContent className="pt-0">
                  <div className="flex items-center justify-between text-sm text-muted-foreground mb-3">
                    <span>{tour.steps} steps</span>
                    <span>{tour.duration}</span>
                  </div>
                  
                  {!progress.isCompleted && (
                    <div className="mb-3">
                      <div className="flex items-center justify-between text-xs text-muted-foreground mb-1">
                        <span>Progress</span>
                        <span>{progress.completed}/{progress.total}</span>
                      </div>
                      <div className="w-full bg-muted rounded-full h-2">
                        <div
                          className="bg-primary h-2 rounded-full transition-all duration-300"
                          style={{
                            width: `${progress.total > 0 ? (progress.completed / progress.total) * 100 : 0}%`
                          }}
                        />
                      </div>
                    </div>
                  )}
                  
                  <div className="flex gap-2">
                    <Button
                      onClick={() => handleStartTour(tourType)}
                      size="sm"
                      className="flex-1 gap-2"
                      variant={progress.isCompleted ? "outline" : "default"}
                    >
                      {progress.isCompleted ? (
                        <>
                          <RotateCcw className="h-3 w-3" />
                          Restart Tutorial
                        </>
                      ) : (
                        <>
                          <Play className="h-3 w-3" />
                          {progress.completed > 0 ? "Continue" : "Start Tutorial"}
                        </>
                      )}
                    </Button>
                  </div>
                </CardContent>
              </Card>
            );
          })}
        </div>
        
        <div className="mt-6 p-4 bg-muted/50 rounded-lg">
          <h4 className="font-medium mb-2 flex items-center gap-2">
            <HelpCircle className="h-4 w-4" />
            Need More Help?
          </h4>
          <p className="text-sm text-muted-foreground mb-3">
            These interactive tutorials will guide you through the most important features of SportsPro.
            You can restart any tutorial at any time.
          </p>
          <div className="flex gap-2 text-xs text-muted-foreground">
            <span>• Click and follow the highlighted areas</span>
            <span>• Use the Back/Next buttons to navigate</span>
            <span>• Press ESC to exit anytime</span>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}