import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { ExpandIcon } from "lucide-react";
import {
  ResponsiveContainer,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  PieChart,
  Pie,
  Cell,
  LineChart,
  Line,
  Legend,
} from "recharts";

interface DashboardChartsProps {
  stats?: {
    playersByTeam: { teamName: string; playerCount: number }[];
    playersByGender: { male: number; female: number };
    playersByStatus: { active: number; inactive: number; suspended: number };
    playersByAge: { ageRange: string; count: number }[];
  };
}

const COLORS = {
  primary: "#1976D2",
  secondary: "#FF6B35",
  success: "#4CAF50",
  warning: "#FF9800",
  error: "#F44336",
};

export default function DashboardCharts({ stats }: DashboardChartsProps) {
  if (!stats) return null;

  // Gender distribution data
  const genderData = [
    { name: "Male", value: stats.playersByGender.male, color: COLORS.primary },
    { name: "Female", value: stats.playersByGender.female, color: COLORS.secondary },
  ];

  // Status distribution data
  const statusData = [
    { name: "Active", value: stats.playersByStatus.active, color: COLORS.success },
    { name: "Inactive", value: stats.playersByStatus.inactive, color: COLORS.error },
    { name: "Suspended", value: stats.playersByStatus.suspended, color: COLORS.warning },
  ];

  // Age distribution data for line chart
  const ageData = [
    { ageRange: "16-18", male: 45, female: 38 },
    { ageRange: "19-21", male: 78, female: 62 },
    { ageRange: "22-24", male: 95, female: 74 },
    { ageRange: "25-27", male: 67, female: 52 },
    { ageRange: "28-30", male: 34, female: 28 },
    { ageRange: "31+", male: 23, female: 18 },
  ];

  return (
    <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
      {/* Players by Team Chart */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle>Players by Team</CardTitle>
            <Button variant="ghost" size="sm">
              <ExpandIcon className="h-4 w-4" />
            </Button>
          </div>
        </CardHeader>
        <CardContent>
          <ResponsiveContainer width="100%" height={300}>
            <BarChart data={stats.playersByTeam}>
              <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
              <XAxis 
                dataKey="teamName" 
                tick={{ fontSize: 12 }}
                angle={-45}
                textAnchor="end"
                height={80}
              />
              <YAxis tick={{ fontSize: 12 }} />
              <Tooltip />
              <Bar 
                dataKey="playerCount" 
                fill={COLORS.primary}
                radius={[4, 4, 0, 0]}
                name="Players"
              />
            </BarChart>
          </ResponsiveContainer>
        </CardContent>
      </Card>

      {/* Gender Distribution Chart */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle>Gender Distribution</CardTitle>
            <Button variant="ghost" size="sm">
              <ExpandIcon className="h-4 w-4" />
            </Button>
          </div>
        </CardHeader>
        <CardContent>
          <ResponsiveContainer width="100%" height={300}>
            <PieChart>
              <Pie
                data={genderData}
                cx="50%"
                cy="50%"
                innerRadius={60}
                outerRadius={120}
                paddingAngle={5}
                dataKey="value"
              >
                {genderData.map((entry, index) => (
                  <Cell key={`cell-${index}`} fill={entry.color} />
                ))}
              </Pie>
              <Tooltip />
              <Legend />
            </PieChart>
          </ResponsiveContainer>
        </CardContent>
      </Card>

      {/* Age Distribution Chart */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle>Age Distribution</CardTitle>
            <Select defaultValue="all">
              <SelectTrigger className="w-32">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Teams</SelectItem>
                {stats.playersByTeam.map((team) => (
                  <SelectItem key={team.teamName} value={team.teamName}>
                    {team.teamName}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </CardHeader>
        <CardContent>
          <ResponsiveContainer width="100%" height={300}>
            <LineChart data={ageData}>
              <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
              <XAxis dataKey="ageRange" tick={{ fontSize: 12 }} />
              <YAxis tick={{ fontSize: 12 }} />
              <Tooltip />
              <Legend />
              <Line 
                type="monotone" 
                dataKey="male" 
                stroke={COLORS.primary}
                strokeWidth={3}
                dot={{ fill: COLORS.primary, strokeWidth: 2, r: 4 }}
                name="Male"
              />
              <Line 
                type="monotone" 
                dataKey="female" 
                stroke={COLORS.secondary}
                strokeWidth={3}
                dot={{ fill: COLORS.secondary, strokeWidth: 2, r: 4 }}
                name="Female"
              />
            </LineChart>
          </ResponsiveContainer>
        </CardContent>
      </Card>

      {/* Player Status Chart */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle>Player Status</CardTitle>
            <div className="flex space-x-2">
              <Badge className="bg-success text-white">Active</Badge>
              <Badge variant="destructive">Inactive</Badge>
              <Badge className="bg-warning text-white">Suspended</Badge>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          <ResponsiveContainer width="100%" height={300}>
            <PieChart>
              <Pie
                data={statusData}
                cx="50%"
                cy="50%"
                outerRadius={120}
                dataKey="value"
                label={({ name, percent }) => `${name} ${(percent * 100).toFixed(0)}%`}
              >
                {statusData.map((entry, index) => (
                  <Cell key={`cell-${index}`} fill={entry.color} />
                ))}
              </Pie>
              <Tooltip />
            </PieChart>
          </ResponsiveContainer>
        </CardContent>
      </Card>
    </div>
  );
}
