import { useEffect } from 'react';
import { useLocation } from 'wouter';

interface SEOProps {
  title?: string;
  description?: string;
  keywords?: string;
  image?: string;
  url?: string;
  type?: string;
}

export default function SEOHead({
  title = "SportsPro - Professional Sports Management Platform",
  description = "Comprehensive sports management platform for teams and athletes. Track performance, manage rosters, schedule training sessions, and analyze data with advanced analytics tools.",
  keywords = "sports management, team management, athlete tracking, performance analytics, roster management, training schedule, sports platform, rugby, soccer, basketball, cricket",
  image = "/assets/sportspro-og-image.jpg",
  url,
  type = "website"
}: SEOProps) {
  const [location] = useLocation();
  
  const currentUrl = url || `${window.location.origin}${location}`;
  
  useEffect(() => {
    // Update document title
    document.title = title;
    
    // Update or create meta tags
    const updateMetaTag = (name: string, content: string, property?: boolean) => {
      const selector = property ? `meta[property="${name}"]` : `meta[name="${name}"]`;
      let meta = document.querySelector(selector) as HTMLMetaElement;
      
      if (!meta) {
        meta = document.createElement('meta');
        if (property) {
          meta.setAttribute('property', name);
        } else {
          meta.setAttribute('name', name);
        }
        document.head.appendChild(meta);
      }
      
      meta.setAttribute('content', content);
    };
    
    // Basic meta tags
    updateMetaTag('description', description);
    updateMetaTag('keywords', keywords);
    
    // Open Graph tags
    updateMetaTag('og:title', title, true);
    updateMetaTag('og:description', description, true);
    updateMetaTag('og:type', type, true);
    updateMetaTag('og:url', currentUrl, true);
    updateMetaTag('og:image', image, true);
    updateMetaTag('og:site_name', 'SportsPro', true);
    
    // Twitter Card tags
    updateMetaTag('twitter:card', 'summary_large_image');
    updateMetaTag('twitter:title', title);
    updateMetaTag('twitter:description', description);
    updateMetaTag('twitter:image', image);
    
    // Update canonical URL
    let canonical = document.querySelector('link[rel="canonical"]') as HTMLLinkElement;
    if (!canonical) {
      canonical = document.createElement('link');
      canonical.setAttribute('rel', 'canonical');
      document.head.appendChild(canonical);
    }
    canonical.setAttribute('href', currentUrl);
    
  }, [title, description, keywords, image, currentUrl, type]);
  
  return null;
}

// Page-specific SEO configurations
export const seoConfigs = {
  landing: {
    title: "SportsPro - Professional Sports Management Platform",
    description: "Transform your sports organization with comprehensive team management, player tracking, performance analytics, and scheduling tools. Support for 10+ sports including Rugby, Soccer, Basketball, and Cricket.",
    keywords: "sports management software, team management platform, athlete performance tracking, sports analytics, roster management, training schedule, multi-sport platform"
  },
  
  dashboard: {
    title: "Dashboard - SportsPro",
    description: "Sports management dashboard with real-time analytics, team performance metrics, and organizational insights for coaches and administrators.",
    keywords: "sports dashboard, team analytics, performance metrics, sports statistics"
  },
  
  teams: {
    title: "Team Management - SportsPro",
    description: "Create and manage sports teams with roster organization, player assignments, coaching staff management, and team performance tracking.",
    keywords: "team management, roster management, sports teams, team organization, coaching staff"
  },
  
  players: {
    title: "Player Management - SportsPro",
    description: "Comprehensive player profiles with performance tracking, attendance monitoring, skill development, and detailed statistics across multiple sports.",
    keywords: "player management, athlete profiles, player statistics, performance tracking, sports players"
  },
  
  analytics: {
    title: "Sports Analytics - SportsPro",
    description: "Advanced sports analytics with performance insights, trend analysis, comparative statistics, and data-driven decision making tools for teams and athletes.",
    keywords: "sports analytics, performance analysis, sports statistics, data analytics, sports insights"
  },
  
  attendance: {
    title: "Attendance Tracking - SportsPro",
    description: "Track training session attendance, monitor player participation, generate attendance reports, and automate notifications for sports organizations.",
    keywords: "attendance tracking, training attendance, session monitoring, sports attendance, player participation"
  },
  
  coaches: {
    title: "Coach Management - SportsPro",
    description: "Manage coaching staff with detailed profiles, certifications, experience tracking, and team assignments for professional sports organizations.",
    keywords: "coach management, coaching staff, sports coaches, team coaching, coach profiles"
  },
  
  tournaments: {
    title: "Tournament Management - SportsPro",
    description: "Organize and manage sports tournaments with bracket generation, match scheduling, results tracking, and championship administration.",
    keywords: "tournament management, sports tournaments, competition organization, match scheduling, sports events"
  },
  
  performance: {
    title: "Performance Tracking - SportsPro",
    description: "Individual and team performance monitoring with detailed metrics, progress tracking, goal setting, and achievement analysis for athletes.",
    keywords: "performance tracking, athlete performance, sports metrics, performance analysis, sports performance"
  },
  
  training: {
    title: "Training Management - SportsPro",
    description: "Plan and manage training sessions with scheduling tools, drill libraries, progress tracking, and customized training programs for sports teams.",
    keywords: "training management, sports training, training schedules, workout planning, sports drills"
  }
};