# SportsPro Platform - Quick Deployment Guide

## 🚀 Quick Start Commands

### VPS Deployment (Ubuntu 20.04+)
```bash
# 1. System Setup
sudo apt update && sudo apt upgrade -y
sudo apt install -y curl wget git nginx postgresql postgresql-contrib

# 2. Install Node.js 20.x
curl -fsSL https://deb.nodesource.com/setup_20.x | sudo -E bash -
sudo apt-get install -y nodejs

# 3. Install PM2
sudo npm install -g pm2

# 4. Clone and Setup
git clone <your-repo-url> sports-platform
cd sports-platform
npm install
npm run build

# 5. Database Setup
sudo -u postgres createdb sports_platform
sudo -u postgres createuser sportsapp -P

# 6. Environment Configuration
cp .env.example .env.production
# Edit .env.production with your settings

# 7. Database Migration
npm run db:push

# 8. Start Application
pm2 start ecosystem.config.js
pm2 save
```

### Required Environment Variables
```env
NODE_ENV=production
PORT=5000
DATABASE_URL=postgresql://username:password@localhost:5432/sports_platform
SESSION_SECRET=your_very_long_random_session_secret
REPL_ID=your_replit_app_id
REPLIT_DOMAINS=yourdomain.com
```

### Nginx Configuration
```nginx
server {
    listen 443 ssl http2;
    server_name yourdomain.com;
    
    ssl_certificate /etc/letsencrypt/live/yourdomain.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/yourdomain.com/privkey.pem;
    
    location / {
        proxy_pass http://localhost:5000;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
}
```

### SSL Certificate (Let's Encrypt)
```bash
sudo apt install certbot python3-certbot-nginx
sudo certbot --nginx -d yourdomain.com
```

## 📋 Shared Hosting Deployment

### cPanel Steps
1. **Upload Files:** Extract application to `public_html`
2. **Node.js Setup:** Enable Node.js 18.x+ in cPanel
3. **Database:** Create PostgreSQL/MySQL database
4. **Environment:** Configure variables in Node.js section
5. **Dependencies:** Run `npm install --production`
6. **Migration:** Execute `npm run db:push`
7. **Start:** Launch application through cPanel

## 🔧 Essential Commands

### PM2 Management
```bash
pm2 start sports-platform    # Start application
pm2 restart sports-platform  # Restart application
pm2 logs sports-platform     # View logs
pm2 monit                    # Monitor resources
pm2 save                     # Save configuration
```

### Database Operations
```bash
npm run db:push              # Apply schema changes
npm run db:seed              # Create sample data
pg_dump $DATABASE_URL > backup.sql  # Backup database
```

### Monitoring
```bash
pm2 logs --lines 50          # Recent logs
sudo nginx -t                # Test Nginx config
curl http://localhost:5000   # Test application
```

## ⚡ Performance Tips

1. **Enable Nginx caching for static assets**
2. **Use PM2 cluster mode for multiple CPU cores**
3. **Set up log rotation with PM2**
4. **Configure database connection pooling**
5. **Enable gzip compression in Nginx**

## 🔒 Security Checklist

- [ ] Strong session secret (64+ characters)
- [ ] Database user with limited privileges
- [ ] SSL certificate installed and configured
- [ ] Firewall configured (UFW/iptables)
- [ ] Regular security updates scheduled
- [ ] Environment files secured (chmod 600)
- [ ] Fail2ban configured for SSH protection

## 📞 Support

For detailed instructions, see [INSTALLATION_GUIDE.md](./INSTALLATION_GUIDE.md)

### Common Issues
- **Port 5000 in use:** Change PORT in environment
- **Database connection failed:** Check DATABASE_URL format
- **Nginx 502 error:** Verify application is running on correct port
- **File upload issues:** Check uploads directory permissions

### Health Check
```bash
curl -f http://localhost:5000/api/health || echo "Application down"
```

### Quick Troubleshooting
```bash
# Check application status
pm2 status

# View recent errors
pm2 logs sports-platform --err --lines 20

# Restart if needed
pm2 restart sports-platform

# Check database connection
psql $DATABASE_URL -c "SELECT NOW();"
```