# SportsPro Deployment Guide

## Complete Self-Hosting Installation for Custom Domains

This guide covers deploying SportsPro on your own domain with a VPS or dedicated server.

## Prerequisites

- Ubuntu 20.04+ or CentOS 8+ server
- Domain name with DNS access
- Minimum 2GB RAM, 2 CPU cores, 20GB storage
- Root or sudo access

## Quick Start

```bash
# Download and run the automated installer
curl -fsSL https://raw.githubusercontent.com/sportspro/install/main/install.sh | bash
```

## Manual Installation

### 1. Server Setup

```bash
# Update system
sudo apt update && sudo apt upgrade -y

# Install Node.js 20.x
curl -fsSL https://deb.nodesource.com/setup_20.x | sudo -E bash -
sudo apt-get install -y nodejs

# Install PostgreSQL
sudo apt install postgresql postgresql-contrib -y

# Install Nginx
sudo apt install nginx -y

# Install PM2 globally
sudo npm install -g pm2

# Install SSL certificate tool
sudo snap install --classic certbot
sudo ln -s /snap/bin/certbot /usr/bin/certbot
```

### 2. Database Configuration

```bash
# Switch to postgres user
sudo -u postgres psql

# Create database and user
CREATE DATABASE sportspro;
CREATE USER sportspro_user WITH PASSWORD 'your_secure_password';
GRANT ALL PRIVILEGES ON DATABASE sportspro TO sportspro_user;
\q
```

### 3. Application Setup

```bash
# Create application directory
sudo mkdir -p /var/www/sportspro
cd /var/www/sportspro

# Clone repository (replace with your repo URL)
git clone https://github.com/your-repo/sportspro.git .

# Install dependencies
npm install

# Build application
npm run build
```

### 4. Environment Configuration

Create `.env` file:

```bash
sudo nano /var/www/sportspro/.env
```

Add the following configuration:

```env
# Database Configuration
DATABASE_URL=postgresql://sportspro_user:your_secure_password@localhost:5432/sportspro

# Session Configuration
SESSION_SECRET=your_super_secure_session_secret_minimum_32_characters

# Email Configuration (SMTP)
SMTP_HOST=smtp.gmail.com
SMTP_PORT=587
SMTP_USERNAME=your-email@gmail.com
SMTP_PASSWORD=your-app-password
FROM_EMAIL=noreply@yourdomain.com
FROM_NAME=SportsPro Platform
REPLY_TO_EMAIL=support@yourdomain.com

# Application Configuration
NODE_ENV=production
PORT=3000
DOMAIN=yourdomain.com
BASE_URL=https://yourdomain.com

# Optional: Analytics and Integrations
GOOGLE_ANALYTICS_ID=GA_MEASUREMENT_ID
STRIPE_SECRET_KEY=sk_live_your_stripe_secret
STRIPE_PUBLISHABLE_KEY=pk_live_your_stripe_publishable
```

### 5. Database Migration

```bash
# Run database migrations
npm run db:push

# Seed initial data (optional)
npm run seed
```

### 6. PM2 Process Management

Create `ecosystem.config.js`:

```javascript
module.exports = {
  apps: [{
    name: 'sportspro',
    script: 'server/index.js',
    instances: 'max',
    exec_mode: 'cluster',
    env: {
      NODE_ENV: 'production',
      PORT: 3000
    },
    error_file: '/var/log/sportspro/error.log',
    out_file: '/var/log/sportspro/out.log',
    log_file: '/var/log/sportspro/combined.log',
    time: true
  }]
};
```

Start the application:

```bash
# Create log directory
sudo mkdir -p /var/log/sportspro
sudo chown -R $USER:$USER /var/log/sportspro

# Start with PM2
pm2 start ecosystem.config.js
pm2 save
pm2 startup
```

### 7. Nginx Configuration

Create Nginx configuration:

```bash
sudo nano /etc/nginx/sites-available/sportspro
```

Add the following:

```nginx
server {
    listen 80;
    server_name yourdomain.com www.yourdomain.com;
    
    # Redirect HTTP to HTTPS
    return 301 https://$server_name$request_uri;
}

server {
    listen 443 ssl http2;
    server_name yourdomain.com www.yourdomain.com;
    
    # SSL Configuration (will be managed by Certbot)
    ssl_certificate /etc/letsencrypt/live/yourdomain.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/yourdomain.com/privkey.pem;
    include /etc/letsencrypt/options-ssl-nginx.conf;
    ssl_dhparam /etc/letsencrypt/ssl-dhparams.pem;
    
    # Security headers
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-Content-Type-Options "nosniff" always;
    add_header X-XSS-Protection "1; mode=block" always;
    add_header Referrer-Policy "no-referrer-when-downgrade" always;
    add_header Content-Security-Policy "default-src 'self' http: https: data: blob: 'unsafe-inline'" always;
    
    # Gzip compression
    gzip on;
    gzip_vary on;
    gzip_min_length 1024;
    gzip_proxied expired no-cache no-store private must-revalidate auth;
    gzip_types text/plain text/css text/xml text/javascript application/x-javascript application/xml+rss application/javascript;
    
    # Static files
    location /assets/ {
        alias /var/www/sportspro/client/dist/assets/;
        expires 1y;
        add_header Cache-Control "public, immutable";
    }
    
    location /uploads/ {
        alias /var/www/sportspro/uploads/;
        expires 30d;
        add_header Cache-Control "public";
    }
    
    # API requests
    location /api/ {
        proxy_pass http://localhost:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_cache_bypass $http_upgrade;
    }
    
    # Main application
    location / {
        proxy_pass http://localhost:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_cache_bypass $http_upgrade;
        
        # Fallback for React Router
        try_files $uri $uri/ @fallback;
    }
    
    location @fallback {
        proxy_pass http://localhost:3000;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
}
```

Enable the site:

```bash
sudo ln -s /etc/nginx/sites-available/sportspro /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl reload nginx
```

### 8. SSL Certificate Setup

```bash
# Install SSL certificate with Certbot
sudo certbot --nginx -d yourdomain.com -d www.yourdomain.com

# Test automatic renewal
sudo certbot renew --dry-run
```

### 9. Firewall Configuration

```bash
# Configure UFW firewall
sudo ufw allow OpenSSH
sudo ufw allow 'Nginx Full'
sudo ufw --force enable
```

### 10. Monitoring and Logs

Setup log rotation:

```bash
sudo nano /etc/logrotate.d/sportspro
```

Add:

```
/var/log/sportspro/*.log {
    daily
    missingok
    rotate 14
    compress
    notifempty
    create 644 $USER $USER
    postrotate
        pm2 reloadLogs
    endscript
}
```

## Email Provider Setup

### Gmail Setup

1. Enable 2-Factor Authentication
2. Generate App Password:
   - Go to Google Account settings
   - Security → 2-Step Verification → App passwords
   - Generate password for "Mail"
3. Use app password in SMTP_PASSWORD

### SendGrid Setup

1. Create SendGrid account
2. Generate API key
3. Configure:
   ```env
   SMTP_HOST=smtp.sendgrid.net
   SMTP_PORT=587
   SMTP_USERNAME=apikey
   SMTP_PASSWORD=your_sendgrid_api_key
   ```

### AWS SES Setup

1. Create AWS account and verify domain
2. Create IAM user with SES permissions
3. Configure:
   ```env
   SMTP_HOST=email-smtp.us-east-1.amazonaws.com
   SMTP_PORT=587
   SMTP_USERNAME=your_aws_access_key
   SMTP_PASSWORD=your_aws_secret_key
   ```

## Domain Configuration

### DNS Records

Set up the following DNS records:

```
Type    Name    Value                   TTL
A       @       your_server_ip          300
A       www     your_server_ip          300
CNAME   mail    yourdomain.com          300
TXT     @       "v=spf1 include:_spf.google.com ~all"  300
```

### SSL Certificate Renewal

Certbot automatically sets up renewal. Verify with:

```bash
sudo systemctl status snap.certbot.renew.timer
```

## Backup Strategy

### Database Backup

Create backup script:

```bash
sudo nano /usr/local/bin/backup-sportspro.sh
```

```bash
#!/bin/bash
BACKUP_DIR="/var/backups/sportspro"
DATE=$(date +"%Y%m%d_%H%M%S")

# Create backup directory
mkdir -p $BACKUP_DIR

# Database backup
pg_dump -h localhost -U sportspro_user sportspro > $BACKUP_DIR/db_backup_$DATE.sql

# Application files backup
tar -czf $BACKUP_DIR/app_backup_$DATE.tar.gz /var/www/sportspro --exclude=node_modules

# Keep only last 7 days of backups
find $BACKUP_DIR -name "*.sql" -mtime +7 -delete
find $BACKUP_DIR -name "*.tar.gz" -mtime +7 -delete
```

Make executable and add to cron:

```bash
sudo chmod +x /usr/local/bin/backup-sportspro.sh
sudo crontab -e

# Add this line for daily backups at 2 AM
0 2 * * * /usr/local/bin/backup-sportspro.sh
```

## Troubleshooting

### Common Issues

1. **Application won't start**
   ```bash
   pm2 logs sportspro
   sudo systemctl status nginx
   ```

2. **Database connection errors**
   ```bash
   sudo -u postgres psql -c "\l"
   sudo systemctl status postgresql
   ```

3. **SSL certificate issues**
   ```bash
   sudo certbot certificates
   sudo nginx -t
   ```

4. **Email not working**
   ```bash
   # Test SMTP connection
   telnet smtp.gmail.com 587
   ```

### Performance Optimization

1. **Enable Redis for sessions** (optional)
   ```bash
   sudo apt install redis-server -y
   ```
   
   Update `.env`:
   ```env
   REDIS_URL=redis://localhost:6379
   ```

2. **Database optimization**
   ```bash
   sudo nano /etc/postgresql/12/main/postgresql.conf
   ```
   
   Adjust:
   ```
   shared_buffers = 256MB
   effective_cache_size = 1GB
   max_connections = 100
   ```

3. **Enable HTTP/2 and compression**
   Already configured in Nginx setup above.

## Security Checklist

- [ ] Strong passwords for all accounts
- [ ] Firewall configured (UFW)
- [ ] SSL certificate installed
- [ ] Regular security updates scheduled
- [ ] Database access restricted
- [ ] Application logs monitored
- [ ] Backup strategy implemented
- [ ] Environment variables secured

## Updating SportsPro

```bash
cd /var/www/sportspro
git pull origin main
npm install
npm run build
npm run db:push
pm2 restart sportspro
```

## Support

For deployment issues:
- Check application logs: `pm2 logs sportspro`
- Check system logs: `sudo journalctl -u nginx`
- Verify configuration: `sudo nginx -t`

For questions or support, contact the SportsPro team or refer to the documentation at your deployed instance under `/help`.